<?php
session_start();
include('db.php');
// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
// Get the logged-in user's ID
$currentUserId = $_SESSION['user_id'];
// Fetch the current user's balance
$balanceQuery = $conn->prepare("SELECT balance FROM users WHERE id = ?");
$balanceQuery->bind_param("i", $currentUserId);
$balanceQuery->execute();
$balanceResult = $balanceQuery->get_result();
$userData = $balanceResult->fetch_assoc();
$balance = $userData ? $userData['balance'] : 0;
// Fetch users excluding the current user
$searchQuery = "SELECT id, name, phone FROM users WHERE id != ?";
$searchStmt = $conn->prepare($searchQuery);
$searchStmt->bind_param("i", $currentUserId);
$searchStmt->execute();
$usersResult = $searchStmt->get_result();
$users = [];
while ($row = $usersResult->fetch_assoc()) {
    $users[] = $row;
}
// Function to mask phone number: max 4 '*' and show last 4 digits
function maskPhoneNumber($phone) {
    $visiblePart = substr($phone, -4); // Last 4 digits
    $maskedLength = strlen($phone) - 4; // Number of characters to mask
    if ($maskedLength > 4) {
        $maskedLength = 4; // Limit max '*' to 4
    }
    $maskedPart = str_repeat('*', $maskedLength);
    return $maskedPart . $visiblePart; // Concatenate masked part with last 4 digits
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Search Users by Phone</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #14b8a6;
            --primary-dark: #0d9488;
            --primary-light: #ccfbf1;
            --secondary-color: #0d9488;
            --accent-color: #FFD700;
            --light-bg: #f8f9fa;
            --card-bg: #ffffff;
            --text-primary: #212529;
            --text-secondary: #6c757d;
            --shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: var(--light-bg);
            color: var(--text-primary);
            line-height: 1.6;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            padding: 15px 20px;
            box-shadow: var(--shadow);
            position: sticky;
            top: 0;
            z-index: 100;
        }
        .back-icon {
            display: flex;
            align-items: center;
            cursor: pointer;
            transition: var(--transition);
        }
        .back-icon:hover {
            transform: translateX(-5px);
        }
        .back-icon i {
            font-size: 20px;
            margin-right: 10px;
        }
        .header h3 {
            font-size: 22px;
            font-weight: 600;
            text-align: center;
            flex-grow: 1;
            margin: 0;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.2);
        }
        .balance {
            display: flex;
            align-items: center;
            background-color: rgba(255, 255, 255, 0.2);
            padding: 8px 15px;
            border-radius: 30px;
            font-weight: 600;
            font-size: 16px;
            backdrop-filter: blur(5px);
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .balance i {
            margin-right: 8px;
            color: var(--accent-color);
        }
        .main-container {
            max-width: 800px;
            margin: 25px auto;
            padding: 0 15px;
        }
        .page-title {
            text-align: center;
            margin: 30px 0 20px;
            color: var(--primary-color);
            position: relative;
            font-size: 24px;
            font-weight: 600;
        }
        .page-title::after {
            content: '';
            position: absolute;
            bottom: -10px;
            left: 50%;
            transform: translateX(-50%);
            width: 80px;
            height: 3px;
            background: linear-gradient(to right, var(--primary-color), var(--secondary-color));
            border-radius: 3px;
        }
        .search-container {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 25px;
            box-shadow: var(--shadow);
            position: relative;
        }
        .search-input-wrapper {
            position: relative;
            display: flex;
            align-items: center;
        }
        .search-input-wrapper i {
            position: absolute;
            left: 15px;
            color: var(--text-secondary);
            font-size: 18px;
        }
        .search-container input {
            width: 100%;
            padding: 15px 15px 15px 45px;
            font-size: 16px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            box-sizing: border-box;
            transition: var(--transition);
            background-color: #f9f9f9;
        }
        .search-container input:focus {
            border-color: var(--primary-color);
            background-color: white;
            outline: none;
            box-shadow: 0 0 0 3px rgba(20, 184, 166, 0.1);
        }
        .search-button {
            position: absolute;
            right: 5px;
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            border: none;
            border-radius: 6px;
            padding: 10px 15px;
            cursor: pointer;
            transition: var(--transition);
            box-shadow: 0 2px 5px rgba(20, 184, 166, 0.2);
        }
        .search-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(20, 184, 166, 0.3);
        }
        .users-container {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 20px;
            box-shadow: var(--shadow);
        }
        .section-title {
            font-size: 18px;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 20px;
            display: flex;
            align-items: center;
        }
        .section-title i {
            margin-right: 10px;
            color: var(--secondary-color);
        }
        .user-item {
            display: flex;
            align-items: center;
            padding: 15px;
            margin-bottom: 12px;
            border-radius: 10px;
            background: #f8f9fa;
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
            transition: var(--transition);
            cursor: pointer;
            border-left: 4px solid transparent;
        }
        .user-item:hover {
            transform: translateY(-3px);
            box-shadow: 0 5px 15px rgba(20, 184, 166, 0.1);
            border-left-color: var(--primary-color);
        }
        .user-item:last-child {
            margin-bottom: 0;
        }
        .user-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            display: flex;
            justify-content: center;
            align-items: center;
            color: white;
            font-weight: 600;
            font-size: 20px;
            margin-right: 15px;
            box-shadow: 0 2px 5px rgba(20, 184, 166, 0.2);
        }
        .user-info {
            flex-grow: 1;
        }
        .user-name {
            font-size: 16px;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 5px;
        }
        .user-phone {
            font-size: 14px;
            color: var(--text-secondary);
            display: flex;
            align-items: center;
        }
        .user-phone i {
            margin-right: 5px;
            color: var(--accent-color);
        }
        .send-button {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            border: none;
            border-radius: 20px;
            padding: 8px 15px;
            cursor: pointer;
            transition: var(--transition);
            font-size: 14px;
            font-weight: 500;
            display: flex;
            align-items: center;
            box-shadow: 0 2px 5px rgba(20, 184, 166, 0.2);
        }
        .send-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(20, 184, 166, 0.3);
        }
        .send-button i {
            margin-right: 5px;
        }
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: var(--text-secondary);
        }
        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: var(--primary-color);
        }
        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--text-primary);
        }
        @media (max-width: 768px) {
            .header {
                padding: 12px 15px;
            }
            
            .header h3 {
                font-size: 18px;
            }
            
            .balance {
                font-size: 14px;
                padding: 6px 12px;
            }
            
            .page-title {
                font-size: 20px;
            }
            
            .search-container {
                padding: 15px;
            }
            
            .user-item {
                padding: 12px;
            }
            
            .user-avatar {
                width: 40px;
                height: 40px;
                font-size: 18px;
            }
            
            .user-name {
                font-size: 15px;
            }
            
            .user-phone {
                font-size: 13px;
            }
        }
    </style>
</head>
<body>
    <!-- Header Section -->
    <div class="header">
        <div class="back-icon" onclick="history.back()">
            <i class="fas fa-arrow-left"></i>
        </div>
        <h3>Search Users</h3>
        <div class="balance">
            <i class="fas fa-wallet"></i>
            ৳<?php echo number_format($balance, 2); ?>
        </div>
    </div>
    <!-- Main Container -->
    <div class="main-container">
        <h2 class="page-title">Send Money</h2>
        
        <!-- Search Container -->
        <div class="search-container">
            <div class="search-input-wrapper">
                <i class="fas fa-search"></i>
                <input type="text" id="searchInput" placeholder="Enter phone number to search...">
                <button id="searchButton" onclick="searchUsers()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
        </div>
        
        <!-- Users List Container -->
        <div class="users-container">
            <div class="section-title">
                <i class="fas fa-users"></i>
                Search Results
            </div>
            <div id="usersContainer">
                <!-- Dynamic User List (Initially Empty) -->
                <div class="empty-state">
                    <i class="fas fa-search"></i>
                    <h3>Search for Users</h3>
                    <p>Enter a phone number to find users you can send money to.</p>
                </div>
            </div>
        </div>
    </div>
    <script>
        let users = <?php echo json_encode($users); ?>; // All users data from PHP
        
        function searchUsers() {
            let input = document.getElementById('searchInput').value;
            let filter = input.trim();
            let usersContainer = document.getElementById('usersContainer');
            usersContainer.innerHTML = ''; // Clear previous search results
            
            // Check if the filter is a complete phone number
            if (filter.length >= 10) { // Assuming phone numbers are at least 10 digits
                let foundUsers = users.filter(user => user.phone === filter); // Exact match
                
                if (foundUsers.length > 0) {
                    foundUsers.forEach(function(user) {
                        let userDiv = document.createElement('div');
                        userDiv.className = 'user-item';
                        
                        // Get user's initials for avatar
                        let initials = user.name.split(' ').map(n => n[0]).join('').toUpperCase();
                        if (initials.length > 2) initials = initials.substring(0, 2);
                        
                        userDiv.innerHTML = `
                            <div class="user-avatar">${initials}</div>
                            <div class="user-info">
                                <div class="user-name">${user.name}</div>
                                <div class="user-phone">
                                    <i class="fas fa-phone-alt"></i>
                                    ${maskPhoneNumber(user.phone)}
                                </div>
                            </div>
                            <button class="send-button" onclick="sendProcess(${user.id})">
                                <i class="fas fa-paper-plane"></i>
                                Send
                            </button>
                        `;
                        usersContainer.appendChild(userDiv);
                    });
                } else {
                    usersContainer.innerHTML = `
                        <div class="empty-state">
                            <i class="fas fa-user-slash"></i>
                            <h3>No User Found</h3>
                            <p>No user found with the phone number: ${filter}</p>
                        </div>
                    `;
                }
            } else {
                usersContainer.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-info-circle"></i>
                        <h3>Enter Full Number</h3>
                        <p>Please enter a complete phone number (at least 10 digits) to search for users.</p>
                    </div>
                `;
            }
        }
        
        // Mask the phone number (same logic as PHP)
        function maskPhoneNumber(phone) {
            let visiblePart = phone.slice(-4); // Last 4 digits
            let maskedLength = phone.length - 4; // Number of characters to mask
            if (maskedLength > 4) {
                maskedLength = 4; // Limit max '*' to 4
            }
            let maskedPart = '*'.repeat(maskedLength);
            return maskedPart + visiblePart; // Concatenate masked part with last 4 digits
        }
        
        // Redirect to send-process.php with the selected user's ID
        function sendProcess(userId) {
            window.location.href = "send-process.php?user_id=" + userId;
        }
        
        // Allow search on Enter key press
        document.getElementById('searchInput').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                searchUsers();
            }
        });
    </script>
</body>
</html>