<?php
session_start();
include('db.php'); // Your database connection
// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$userId = $_SESSION['user_id']; // Current logged-in user
// Fetch user data (name, phone, balance, profile image)
$userQuery = $conn->prepare("SELECT name, phone, balance, profile_image, pin FROM users WHERE id = ?");
$userQuery->bind_param("i", $userId);
$userQuery->execute();
$userResult = $userQuery->get_result();
$userData = $userResult->fetch_assoc();
// Handle profile image upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['profile_image'])) {
    $image = $_FILES['profile_image'];
    
    // Image upload handling (e.g., moving to a directory and updating in DB)
    if ($image['error'] === 0) {
        $imageName = $userId . "_" . basename($image['name']); // Unique image name
        $imagePath = "img/uploads/" . $imageName;
        move_uploaded_file($image['tmp_name'], $imagePath); // Move uploaded file
        // Update the profile image in the database
        $updateImageQuery = $conn->prepare("UPDATE users SET profile_image = ? WHERE id = ?");
        $updateImageQuery->bind_param("si", $imagePath, $userId);
        $updateImageQuery->execute();
        // Reload page to show updated image
        header("Location: " . $_SERVER['PHP_SELF']);
        exit;
    }
}
$pinError = '';
$passwordError = '';
$successMessage = '';
// Handle PIN and password changes (same as before)
// Handle PIN change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['pin_change'])) {
    $oldPin = $_POST['old_pin'];
    $newPin = $_POST['new_pin'];
    if ($oldPin != $userData['pin']) {
        $pinError = "Incorrect old PIN!";
    } else {
        // Update the new PIN in the database
        $updatePinQuery = $conn->prepare("UPDATE users SET pin = ? WHERE id = ?");
        $updatePinQuery->bind_param("si", $newPin, $userId);
        $updatePinQuery->execute();
        $successMessage = "PIN changed successfully!";
    }
}
// Handle Password change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['password_change'])) {
    $oldPassword = $_POST['old_password'];
    $newPassword = $_POST['new_password'];
    // Assuming the password is stored hashed (example using bcrypt)
    $passwordQuery = $conn->prepare("SELECT password FROM users WHERE id = ?");
    $passwordQuery->bind_param("i", $userId);
    $passwordQuery->execute();
    $passwordResult = $passwordQuery->get_result();
    $passwordData = $passwordResult->fetch_assoc();
    if (!password_verify($oldPassword, $passwordData['password'])) {
        $passwordError = "Incorrect old password!";
    } else {
        // Hash the new password and update it
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $updatePasswordQuery = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
        $updatePasswordQuery->bind_param("si", $hashedPassword, $userId);
        $updatePasswordQuery->execute();
        $successMessage = "Password changed successfully!";
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Profile</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #008080; /* Changed to teal */
            --secondary-color: #4dbdbd; /* Lighter teal for secondary elements */
            --accent-color: #FFD700; /* Kept gold accent */
            --light-bg: #f0f8f8; /* Very light teal background */
            --card-bg: #ffffff;
            --text-primary: #004d40; /* Dark teal text */
            --text-secondary: #00695c; /* Medium teal text */
            --success-color: #2ecc71; /* Success color (unchanged) */
            --error-color: #e74c3c; /* Error color (unchanged) */
            --shadow: 0 4px 12px rgba(0, 128, 128, 0.15); /* Teal shadow */
            --transition: all 0.3s ease;
        }
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: var(--light-bg);
            color: var(--text-primary);
            line-height: 1.6;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color)); /* Teal gradient */
            color: white;
            padding: 15px 20px;
            box-shadow: var(--shadow);
            position: sticky;
            top: 0;
            z-index: 100;
        }
        .back-icon {
            display: flex;
            align-items: center;
            cursor: pointer;
            transition: var(--transition);
        }
        .back-icon:hover {
            transform: translateX(-5px);
        }
        .back-icon i {
            font-size: 20px;
            margin-right: 10px;
        }
        .header h3 {
            font-size: 22px;
            font-weight: 600;
            text-align: center;
            flex-grow: 1;
            margin: 0;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.2);
        }
        .balance {
            display: flex;
            align-items: center;
            background-color: rgba(255, 255, 255, 0.2);
            padding: 8px 15px;
            border-radius: 30px;
            font-weight: 600;
            font-size: 16px;
            backdrop-filter: blur(5px);
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .balance i {
            margin-right: 8px;
            color: var(--accent-color);
        }
        .main-container {
            max-width: 800px;
            margin: 25px auto;
            padding: 0 15px;
        }
        .profile-container {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 25px;
            box-shadow: var(--shadow);
            display: flex;
            align-items: center;
            position: relative;
            overflow: hidden;
        }
        .profile-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 5px;
            background: linear-gradient(to right, var(--primary-color), var(--secondary-color)); /* Teal gradient */
        }
        .profile-image-wrapper {
            position: relative;
            margin-right: 20px;
        }
        .profile-image {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            object-fit: cover;
            border: 3px solid var(--primary-color);
            cursor: pointer;
            transition: var(--transition);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        .profile-image:hover {
            transform: scale(1.05);
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
        .camera-icon {
            position: absolute;
            bottom: 0;
            right: 0;
            background: var(--primary-color);
            color: white;
            width: 28px;
            height: 28px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            border: 2px solid white;
            cursor: pointer;
            transition: var(--transition);
        }
        .camera-icon:hover {
            background: var(--secondary-color);
        }
        .profile-info {
            flex-grow: 1;
        }
        .profile-name {
            font-size: 22px;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 5px;
        }
        .profile-phone {
            font-size: 16px;
            color: var(--text-secondary);
            display: flex;
            align-items: center;
        }
        .profile-phone i {
            margin-right: 8px;
            color: var(--accent-color);
        }
        .settings-container {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 25px;
            box-shadow: var(--shadow);
            margin-bottom: 25px;
        }
        .section-title {
            font-size: 18px;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 20px;
            display: flex;
            align-items: center;
        }
        .section-title i {
            margin-right: 10px;
            color: var(--secondary-color);
        }
        .setting-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px;
            margin-bottom: 10px;
            border-radius: 8px;
            transition: var(--transition);
            cursor: pointer;
        }
        .setting-item:hover {
            background-color: rgba(0, 128, 128, 0.05); /* Teal hover background */
        }
        .setting-item:last-child {
            margin-bottom: 0;
        }
        .setting-info {
            display: flex;
            align-items: center;
        }
        .setting-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: rgba(0, 128, 128, 0.1); /* Light teal background */
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 15px;
            color: var(--primary-color);
            font-size: 18px;
        }
        .setting-details {
            flex-grow: 1;
        }
        .setting-title {
            font-size: 16px;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 3px;
        }
        .setting-desc {
            font-size: 14px;
            color: var(--text-secondary);
        }
        .setting-arrow {
            color: var(--text-secondary);
            transition: var(--transition);
        }
        .setting-item:hover .setting-arrow {
            color: var(--primary-color);
            transform: translateX(5px);
        }
        .logout-item .setting-icon {
            background-color: rgba(220, 53, 69, 0.1);
            color: var(--error-color);
        }
        .logout-item .setting-title {
            color: var(--error-color);
        }
        .button-container {
            display: flex;
            justify-content: space-between;
            gap: 15px;
            margin-top: 25px;
        }
        .action-btn {
            flex: 1;
            padding: 15px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        .download-btn {
            background: linear-gradient(135deg, var(--primary-color), #006666); /* Darker teal gradient */
            color: white;
        }
        .contact-btn {
            background: linear-gradient(135deg, #1976d2, #0d47a1); /* Blue gradient (unchanged) */
            color: white;
        }
        .action-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
        .action-btn i {
            margin-right: 8px;
        }
        .popup-overlay {
            visibility: hidden;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            display: flex;
            justify-content: center;
            align-items: center;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        .popup-overlay.show {
            visibility: visible;
            opacity: 1;
        }
        .popup {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 30px;
            width: 90%;
            max-width: 400px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
            transform: scale(0.7);
            transition: transform 0.3s ease;
        }
        .popup-overlay.show .popup {
            transform: scale(1);
        }
        .popup h4 {
            margin-top: 0;
            margin-bottom: 20px;
            text-align: center;
            font-size: 22px;
            color: var(--primary-color);
        }
        .popup input {
            width: 100%;
            padding: 15px;
            margin-bottom: 20px;
            font-size: 16px;
            border: 2px solid #b2dfdb; /* Light teal border */
            border-radius: 8px;
            box-sizing: border-box;
            transition: var(--transition);
            background-color: #f0f8f8; /* Very light teal background */
        }
        .popup input:focus {
            border-color: var(--primary-color);
            background-color: white;
            outline: none;
            box-shadow: 0 0 0 3px rgba(0, 128, 128, 0.1); /* Teal focus shadow */
        }
        .verify-btn, .close-btn {
            width: 100%;
            padding: 15px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            margin-bottom: 10px;
        }
        .verify-btn {
            background: linear-gradient(135deg, var(--primary-color), #006666); /* Darker teal gradient */
            color: white;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        .verify-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
        .close-btn {
            background: linear-gradient(135deg, var(--secondary-color), #006666); /* Teal gradient instead of red */
            color: white;
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        .close-btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
        .message {
            padding: 12px 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-weight: 500;
            display: flex;
            align-items: center;
        }
        .message i {
            margin-right: 10px;
        }
        .error {
            background-color: rgba(220, 53, 69, 0.1);
            color: var(--error-color);
            border-left: 4px solid var(--error-color);
        }
        .success {
            background-color: rgba(40, 167, 69, 0.1);
            color: var(--success-color);
            border-left: 4px solid var(--success-color);
        }
        .switch {
            position: relative;
            display: inline-block;
            width: 50px;
            height: 25px;
        }
        .switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }
        .slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            border-radius: 25px;
            transition: .4s;
        }
        .slider:before {
            position: absolute;
            content: "";
            height: 19px;
            width: 19px;
            border-radius: 50%;
            left: 3px;
            bottom: 3px;
            background-color: white;
            transition: .4s;
        }
        input:checked + .slider {
            background-color: var(--primary-color); /* Teal color for active switch */
        }
        input:checked + .slider:before {
            transform: translateX(24px);
        }
        @media (max-width: 768px) {
            .header {
                padding: 12px 15px;
            }
            
            .header h3 {
                font-size: 18px;
            }
            
            .balance {
                font-size: 14px;
                padding: 6px 12px;
            }
            
            .profile-container {
                padding: 20px;
            }
            
            .profile-image {
                width: 70px;
                height: 70px;
            }
            
            .profile-name {
                font-size: 18px;
            }
            
            .profile-phone {
                font-size: 14px;
            }
            
            .settings-container {
                padding: 20px;
            }
            
            .setting-item {
                padding: 12px;
            }
            
            .button-container {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <!-- Header Section -->
    <div class="header">
        <div class="back-icon" onclick="history.back()">
            <i class="fas fa-arrow-left"></i>
        </div>
        <h3>User Profile</h3>
        <div class="balance">
            <i class="fas fa-wallet"></i>
            ৳<?php echo number_format($userData['balance'], 2); ?>
        </div>
    </div>
    <!-- Main Container -->
    <div class="main-container">
        <!-- Profile Container -->
        <div class="profile-container">
            <div class="profile-image-wrapper">
                <?php 
                    // Always use the default logo image
                    $profileImage = 'image/logo.jpg';
                ?>
                <img src="<?php echo $profileImage; ?>" alt="User Profile" class="profile-image" id="profileImage" onclick="document.getElementById('profileImageInput').click();">
                <div class="camera-icon" onclick="document.getElementById('profileImageInput').click();">
                    <i class="fas fa-camera"></i>
                </div>
            </div>
            
            <div class="profile-info">
                <div class="profile-name"><?php echo htmlspecialchars($userData['name']); ?></div>
                <div class="profile-phone">
                    <i class="fas fa-phone-alt"></i>
                    <?php echo htmlspecialchars($userData['phone']); ?>
                </div>
            </div>
        </div>
        
        <!-- Hidden file input for uploading profile image -->
        <form method="POST" enctype="multipart/form-data">
            <input type="file" name="profile_image" id="profileImageInput" style="display:none;" onchange="this.form.submit();">
        </form>
        
        <!-- Error / Success Messages -->
        <?php if ($pinError): ?>
            <div class="message error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $pinError; ?>
            </div>
        <?php endif; ?>
        <?php if ($passwordError): ?>
            <div class="message error">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo $passwordError; ?>
            </div>
        <?php endif; ?>
        <?php if ($successMessage): ?>
            <div class="message success">
                <i class="fas fa-check-circle"></i>
                <?php echo $successMessage; ?>
            </div>
        <?php endif; ?>
        
        <!-- Settings Section -->
        <div class="settings-container">
            <div class="section-title">
                <i class="fas fa-cog"></i>
                Account Settings
            </div>
            
            <div class="setting-item" onclick="showPopup('pinPopup')">
                <div class="setting-info">
                    <div class="setting-icon">
                        <i class="fas fa-key"></i>
                    </div>
                    <div class="setting-details">
                        <div class="setting-title">PIN Change</div>
                        <div class="setting-desc">Update your security PIN</div>
                    </div>
                </div>
                <i class="fas fa-chevron-right setting-arrow"></i>
            </div>
            
            <div class="setting-item" onclick="showPopup('passwordPopup')">
                <div class="setting-info">
                    <div class="setting-icon">
                        <i class="fas fa-lock"></i>
                    </div>
                    <div class="setting-details">
                        <div class="setting-title">Password Change</div>
                        <div class="setting-desc">Modify your account password</div>
                    </div>
                </div>
                <i class="fas fa-chevron-right setting-arrow"></i>
            </div>
            
            <div class="setting-item">
                <div class="setting-info">
                    <div class="setting-icon">
                        <i class="fas fa-bell"></i>
                    </div>
                    <div class="setting-details">
                        <div class="setting-title">Notifications</div>
                        <div class="setting-desc">Manage your alert preferences</div>
                    </div>
                </div>
                <label class="switch">
                    <input type="checkbox" checked>
                    <span class="slider"></span>
                </label>
            </div>
            
            <div class="setting-item" onclick="window.location.href='reward_convert.php'">
                <div class="setting-info">
                    <div class="setting-icon">
                        <i class="fas fa-gift"></i>
                    </div>
                    <div class="setting-details">
                        <div class="setting-title">Rewards</div>
                        <div class="setting-desc">Rewards Coin Convert in Balance</div>
                    </div>
                </div>
                <i class="fas fa-chevron-right setting-arrow"></i>
            </div>
            
            <div class="setting-item logout-item" onclick="window.location.href='logout.php'">
                <div class="setting-info">
                    <div class="setting-icon">
                        <i class="fas fa-sign-out-alt"></i>
                    </div>
                    <div class="setting-details">
                        <div class="setting-title">Logout</div>
                        <div class="setting-desc">Sign out of your account</div>
                    </div>
                </div>
                <i class="fas fa-chevron-right setting-arrow"></i>
            </div>
        </div>
        
        <!-- Button Container -->
        <div class="button-container">
            <button class="action-btn download-btn" onclick="window.location.href='/App'">
                <i class="fas fa-download"></i>
                Download App
            </button>
            <button class="action-btn contact-btn" onclick="window.location.href='https://wa.me/+8801796003076'">
                <i class="fas fa-comments"></i>
                Contact Us
            </button>
        </div>
    </div>
    <!-- PIN Change Popup -->
    <div class="popup-overlay" id="pinPopup">
        <div class="popup">
            <h4><i class="fas fa-key"></i> Change PIN</h4>
            <form method="POST">
                <input type="hidden" name="pin_change" value="1">
                <input type="password" name="old_pin" placeholder="Old PIN" required>
                <input type="password" name="new_pin" placeholder="New PIN" required>
                <button type="submit" class="verify-btn">
                    <i class="fas fa-check-circle"></i> Update PIN
                </button>
                <button type="button" class="close-btn" onclick="closePopup('pinPopup')">
                    <i class="fas fa-times-circle"></i> Cancel
                </button>
            </form>
        </div>
    </div>
    
    <!-- Password Change Popup -->
    <div class="popup-overlay" id="passwordPopup">
        <div class="popup">
            <h4><i class="fas fa-lock"></i> Change Password</h4>
            <form method="POST">
                <input type="hidden" name="password_change" value="1">
                <input type="password" name="old_password" placeholder="Old Password" required>
                <input type="password" name="new_password" placeholder="New Password" required>
                <button type="submit" class="verify-btn">
                    <i class="fas fa-check-circle"></i> Update Password
                </button>
                <button type="button" class="close-btn" onclick="closePopup('passwordPopup')">
                    <i class="fas fa-times-circle"></i> Cancel
                </button>
            </form>
        </div>
    </div>
    <script>
        function showPopup(popupId) {
            const popup = document.getElementById(popupId);
            popup.classList.add('show');
        }
        
        function closePopup(popupId) {
            const popup = document.getElementById(popupId);
            popup.classList.remove('show');
        }
        
        // Close popup when clicking outside
        document.querySelectorAll('.popup-overlay').forEach(popup => {
            popup.addEventListener('click', function(e) {
                if (e.target === this) {
                    this.classList.remove('show');
                }
            });
        });
    </script>
</body>
</html>