<?php
session_start();
include('db.php'); // Your database connection
// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$userId = $_SESSION['user_id']; // Current logged-in user
// Fetch user balance
$userQuery = $conn->prepare("SELECT balance FROM users WHERE id = ?");
$userQuery->bind_param("i", $userId);
$userQuery->execute();
$userResult = $userQuery->get_result();
$userData = $userResult->fetch_assoc();
$userBalance = $userData['balance'];
// Fetch all paybills from the database
$paybillsQuery = $conn->prepare("SELECT id, name, description, logo_url FROM paybills");
$paybillsQuery->execute();
$paybillsResult = $paybillsQuery->get_result();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bangladeshi Paybills</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #008080; /* Changed to teal */
            --secondary-color: #4dbdbd; /* Lighter teal for secondary elements */
            --accent-color: #FFD700; /* Kept gold accent */
            --light-bg: #f0f8f8; /* Very light teal background */
            --card-bg: #ffffff;
            --text-primary: #004d40; /* Dark teal for text */
            --text-secondary: #00695c; /* Medium teal for secondary text */
            --shadow: 0 4px 12px rgba(0, 128, 128, 0.15); /* Teal shadow */
            --transition: all 0.3s ease;
        }
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: var(--light-bg);
            color: var(--text-primary);
            line-height: 1.6;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color)); /* Teal gradient */
            color: white;
            padding: 15px 20px;
            box-shadow: var(--shadow);
            position: sticky;
            top: 0;
            z-index: 100;
        }
        .back-icon {
            display: flex;
            align-items: center;
            cursor: pointer;
            transition: var(--transition);
        }
        .back-icon:hover {
            transform: translateX(-5px);
        }
        .back-icon i {
            font-size: 20px;
            margin-right: 10px;
        }
        .header h3 {
            font-size: 22px;
            font-weight: 600;
            text-align: center;
            flex-grow: 1;
            margin: 0;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.2);
        }
        .balance {
            display: flex;
            align-items: center;
            background-color: rgba(255, 255, 255, 0.2);
            padding: 8px 15px;
            border-radius: 30px;
            font-weight: 600;
            font-size: 16px;
            backdrop-filter: blur(5px);
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .balance i {
            margin-right: 8px;
            color: var(--accent-color);
        }
        .main-container {
            max-width: 800px;
            margin: 25px auto;
            padding: 0 15px;
        }
        .page-title {
            text-align: center;
            margin: 30px 0 20px;
            color: var(--primary-color);
            position: relative;
            font-size: 24px;
            font-weight: 600;
        }
        .page-title::after {
            content: '';
            position: absolute;
            bottom: -10px;
            left: 50%;
            transform: translateX(-50%);
            width: 80px;
            height: 3px;
            background: linear-gradient(to right, var(--primary-color), var(--secondary-color)); /* Teal gradient */
            border-radius: 3px;
        }
        .search-container {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 25px;
            box-shadow: var(--shadow);
            position: relative;
        }
        .search-container i {
            position: absolute;
            left: 20px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-secondary);
            font-size: 18px;
        }
        .search-container input {
            width: 100%;
            padding: 15px 15px 15px 45px;
            font-size: 16px;
            border: 2px solid #b2dfdb; /* Light teal border */
            border-radius: 8px;
            box-sizing: border-box;
            transition: var(--transition);
            background-color: #f0f8f8; /* Very light teal background */
        }
        .search-container input:focus {
            border-color: var(--primary-color);
            background-color: white;
            outline: none;
            box-shadow: 0 0 0 3px rgba(0, 128, 128, 0.1); /* Teal focus shadow */
        }
        .paybill-list-container {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        .paybill-item {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 20px;
            box-shadow: var(--shadow);
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            border-left: 4px solid transparent;
        }
        .paybill-item:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 20px rgba(0, 128, 128, 0.2); /* Teal shadow */
            border-left-color: var(--primary-color);
        }
        .paybill-logo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            object-fit: contain;
            background-color: #f0f8f8; /* Very light teal background */
            padding: 10px;
            margin-right: 15px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .paybill-details {
            flex-grow: 1;
        }
        .paybill-name {
            font-size: 18px;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 5px;
        }
        .paybill-description {
            font-size: 14px;
            color: var(--text-secondary);
            display: flex;
            align-items: flex-start;
        }
        .paybill-description i {
            margin-right: 5px;
            color: var(--accent-color);
            margin-top: 3px;
        }
        .arrow-icon {
            color: var(--text-secondary);
            transition: var(--transition);
        }
        .paybill-item:hover .arrow-icon {
            color: var(--primary-color);
            transform: translateX(5px);
        }
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: var(--text-secondary);
            grid-column: 1 / -1;
        }
        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: var(--primary-color);
        }
        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--text-primary);
        }
        @media (max-width: 768px) {
            .header {
                padding: 12px 15px;
            }
            
            .header h3 {
                font-size: 18px;
            }
            
            .balance {
                font-size: 14px;
                padding: 6px 12px;
            }
            
            .page-title {
                font-size: 20px;
            }
            
            .search-container {
                padding: 15px;
            }
            
            .search-container input {
                padding: 12px 12px 12px 40px;
            }
            
            .paybill-list-container {
                grid-template-columns: 1fr;
            }
            
            .paybill-item {
                padding: 15px;
            }
            
            .paybill-logo {
                width: 50px;
                height: 50px;
            }
            
            .paybill-name {
                font-size: 16px;
            }
        }
    </style>
</head>
<body>
    <!-- Header Section -->
    <div class="header">
        <div class="back-icon" onclick="history.back()">
            <i class="fas fa-arrow-left"></i>
        </div>
        <h3>Bangladeshi Paybills</h3>
        <div class="balance">
            <i class="fas fa-wallet"></i>
            ৳<?php echo number_format($userBalance, 2); ?>
        </div>
    </div>
    <!-- Main Container -->
    <div class="main-container">
        <h2 class="page-title">Bill Payment Services</h2>
        
        <!-- Search Container -->
        <div class="search-container">
            <i class="fas fa-search"></i>
            <input type="text" id="searchInput" onkeyup="searchPaybills()" placeholder="Search for paybills...">
        </div>
        
        <!-- Paybill List Container -->
        <div class="paybill-list-container" id="paybillList">
            <?php if ($paybillsResult->num_rows > 0): ?>
                <?php while ($paybill = $paybillsResult->fetch_assoc()): ?>
                    <div class="paybill-item" onclick="window.location.href = 'bill-process.php?id=<?php echo $paybill['id']; ?>'">
                        <img src="<?php echo htmlspecialchars($paybill['logo_url']); ?>" alt="<?php echo htmlspecialchars($paybill['name']); ?>" class="paybill-logo">
                        <div class="paybill-details">
                            <div class="paybill-name"><?php echo htmlspecialchars($paybill['name']); ?></div>
                            <div class="paybill-description">
                                <i class="fas fa-info-circle"></i>
                                <?php echo htmlspecialchars($paybill['description']); ?>
                            </div>
                        </div>
                        <i class="fas fa-chevron-right arrow-icon"></i>
                    </div>
                <?php endwhile; ?>
            <?php else: ?>
                <div class="empty-state">
                    <i class="fas fa-file-invoice-dollar"></i>
                    <h3>No Paybills Available</h3>
                    <p>There are currently no paybill services available. Please check back later.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
    <script>
        function searchPaybills() {
            // Get the search input and filter value
            let input = document.getElementById('searchInput');
            let filter = input.value.toUpperCase();
            let paybillList = document.getElementById('paybillList');
            let paybillItems = paybillList.getElementsByClassName('paybill-item');
            let emptyState = paybillList.getElementsByClassName('empty-state')[0];
            let visibleCount = 0;
            
            // Loop through all paybill items and hide those that don't match the search query
            for (let i = 0; i < paybillItems.length; i++) {
                let paybillName = paybillItems[i].getElementsByClassName('paybill-name')[0];
                let paybillDescription = paybillItems[i].getElementsByClassName('paybill-description')[0];
                let nameText = paybillName.textContent || paybillName.innerText;
                let descText = paybillDescription.textContent || paybillDescription.innerText;
                
                if (nameText.toUpperCase().indexOf(filter) > -1 || descText.toUpperCase().indexOf(filter) > -1) {
                    paybillItems[i].style.display = 'flex';
                    visibleCount++;
                } else {
                    paybillItems[i].style.display = 'none';
                }
            }
            
            // Show or hide empty state based on search results
            if (visibleCount === 0 && filter !== '') {
                if (!document.getElementById('searchEmptyState')) {
                    let searchEmptyState = document.createElement('div');
                    searchEmptyState.id = 'searchEmptyState';
                    searchEmptyState.className = 'empty-state';
                    searchEmptyState.innerHTML = `
                        <i class="fas fa-search"></i>
                        <h3>No Matching Paybills</h3>
                        <p>No paybills found matching your search criteria.</p>
                    `;
                    paybillList.appendChild(searchEmptyState);
                }
            } else {
                let searchEmptyState = document.getElementById('searchEmptyState');
                if (searchEmptyState) {
                    searchEmptyState.remove();
                }
            }
        }
    </script>
</body>
</html>