<?php
session_start();
include('db.php'); // Database connection
// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$userId = $_SESSION['user_id']; // Current logged-in user
// Fetch transaction history from the database
$transactionQuery = $conn->prepare("SELECT * FROM transactions WHERE user_id = ? ORDER BY date DESC");
$transactionQuery->bind_param("i", $userId);
$transactionQuery->execute();
$transactionResult = $transactionQuery->get_result();
$transactions = [];
while ($row = $transactionResult->fetch_assoc()) {
    $transactions[] = $row;
}
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ট্রানজেকশন হিস্টরি</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Hind+Siliguri:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Hind Siliguri', sans-serif;
            -webkit-tap-highlight-color: transparent;
        }
        
        :root {
            --teal: #14b8a6;
            --teal-light: #ccfbf1;
            --teal-dark: #0f766e;
            --bkash-dark: #333333;
            --bkash-gray: #666666;
            --bkash-light-gray: #f5f5f5;
            --bkash-border: #e0e0e0;
            --bkash-success: #2ecc71;
            --bkash-warning: #f39c12;
            --bkash-danger: #e74c3c;
            --bkash-blue: #3498db;
            --transition: all 0.3s ease;
        }
        
        body {
            background: var(--bkash-light-gray);
            color: var(--bkash-dark);
            font-size: 14px;
            line-height: 1.4;
        }
        
        .app-container {
            max-width: 480px;
            margin: 0 auto;
            min-height: 100vh;
            background: #ffffff;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
            padding-bottom: 70px;
        }
        
        /* Header Styles */
        .header {
            background: white;
            padding: 15px;
            color: var(--bkash-dark);
            position: relative;
            border-bottom: 1px solid var(--bkash-border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .back-icon {
            display: flex;
            align-items: center;
            cursor: pointer;
            transition: var(--transition);
            color: var(--teal);
        }
        
        .back-icon:hover {
            transform: translateX(-3px);
        }
        
        .back-icon i {
            font-size: 18px;
            margin-right: 8px;
        }
        
        .header h3 {
            font-size: 18px;
            font-weight: 600;
            text-align: center;
            flex-grow: 1;
            margin: 0;
        }
        
        .balance {
            display: flex;
            align-items: center;
            background: var(--teal-light);
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: 600;
            font-size: 14px;
            color: var(--teal);
        }
        
        .balance i {
            margin-right: 6px;
        }
        
        /* Main Container */
        .main-container {
            padding: 15px;
        }
        
        .page-title {
            text-align: center;
            margin: 10px 0 15px;
            color: var(--teal);
            font-size: 18px;
            font-weight: 600;
        }
        
        /* Filter Buttons */
        .button-container {
            display: flex;
            justify-content: space-between;
            margin-bottom: 15px;
            gap: 8px;
        }
        
        .filter-btn {
            flex: 1;
            padding: 8px 5px;
            font-size: 12px;
            font-weight: 500;
            background: white;
            color: var(--teal);
            border: 1px solid var(--teal);
            border-radius: 20px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: var(--transition);
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
        }
        
        .filter-btn:hover {
            background: var(--teal);
            color: white;
            transform: translateY(-2px);
        }
        
        .filter-btn.active {
            background: var(--teal);
            color: white;
        }
        
        .filter-btn i {
            margin-right: 4px;
            font-size: 12px;
        }
        
        /* History Container */
        .history-container {
            background: white;
            border-radius: 8px;
            padding: 10px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }
        
        /* Transaction Item */
        .history-item {
            padding: 10px;
            margin-bottom: 8px;
            border-radius: 8px;
            background: white;
            border: 1px solid var(--bkash-border);
            transition: var(--transition);
            position: relative;
        }
        
        .history-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(20, 184, 166, 0.1);
            border-color: var(--teal);
        }
        
        /* Transaction Header */
        .transaction-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 6px;
        }
        
        .transaction-title {
            font-size: 14px;
            font-weight: 600;
            color: var(--bkash-dark);
            display: flex;
            align-items: center;
        }
        
        .transaction-title i {
            margin-right: 6px;
            font-size: 14px;
            color: var(--teal);
        }
        
        .transaction-date {
            font-size: 11px;
            color: var(--bkash-gray);
        }
        
        /* Transaction Details */
        .transaction-details {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 6px;
            margin-bottom: 6px;
        }
        
        .detail-item {
            display: flex;
            flex-direction: column;
        }
        
        .detail-label {
            font-size: 10px;
            color: var(--bkash-gray);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 2px;
        }
        
        .detail-value {
            font-size: 12px;
            font-weight: 500;
            color: var(--bkash-dark);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        /* Amount and Status Row */
        .amount-status-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 4px;
        }
        
        .transaction-amount {
            font-size: 16px;
            font-weight: 700;
            display: flex;
            align-items: center;
        }
        
        .amount-add {
            color: var(--bkash-success);
        }
        
        .amount-deduct {
            color: var(--bkash-danger);
        }
        
        .transaction-amount i {
            margin-right: 4px;
            font-size: 12px;
        }
        
        .status-badge {
            display: inline-flex;
            align-items: center;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 500;
        }
        
        .status-badge i {
            margin-right: 3px;
            font-size: 6px;
        }
        
        .status-completed {
            background-color: rgba(46, 204, 113, 0.1);
            color: var(--bkash-success);
        }
        
        .status-pending {
            background-color: rgba(243, 156, 18, 0.1);
            color: var(--bkash-warning);
        }
        
        .status-failed {
            background-color: rgba(231, 76, 60, 0.1);
            color: var(--bkash-danger);
        }
        
        /* Transaction Note */
        .transaction-note {
            font-size: 11px;
            color: var(--bkash-gray);
            padding: 4px 8px;
            background-color: var(--teal-light);
            border-radius: 4px;
            display: flex;
            align-items: center;
        }
        
        .transaction-note i {
            margin-right: 4px;
            font-size: 10px;
            color: var(--teal);
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 30px 15px;
            color: var(--bkash-gray);
        }
        
        .empty-state i {
            font-size: 36px;
            margin-bottom: 10px;
            color: var(--teal);
        }
        
        .empty-state h3 {
            font-size: 16px;
            margin-bottom: 5px;
            color: var(--bkash-dark);
        }
        
        .empty-state p {
            font-size: 13px;
        }
        
        /* Bottom Navigation */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            box-shadow: 0 -2px 8px rgba(0, 0, 0, 0.1);
            z-index: 100;
            border-top: 1px solid var(--bkash-border);
        }
        
        .nav-list {
            display: flex;
            justify-content: space-around;
            padding: 8px 0;
        }
        
        .nav-item {
            text-align: center;
            text-decoration: none;
            color: var(--bkash-gray);
            transition: var(--transition);
            padding: 5px;
            border-radius: 8px;
            min-width: 60px;
        }
        
        .nav-item.active {
            color: var(--teal);
        }
        
        .nav-item.active::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 50%;
            transform: translateX(-50%);
            width: 30px;
            height: 3px;
            background: var(--teal);
            border-radius: 3px;
        }
        
        .nav-icon {
            font-size: 18px;
            margin-bottom: 2px;
        }
        
        .nav-label {
            font-size: 10px;
            font-weight: 500;
        }
    </style>
</head>
<body>
    <div class="app-container">
        <!-- Header Section -->
        <div class="header">
            <div class="back-icon" onclick="history.back()">
                <i class="fas fa-arrow-left"></i>
                ফিরে যান
            </div>
            <h3>ট্রানজেকশন হিস্টরি</h3>
            <div class="balance">
                <i class="fas fa-wallet"></i>
                রমিত ওয়ালেট
            </div>
        </div>
        
        <!-- Main Container -->
        <div class="main-container">
            <h2 class="page-title">আপনার ট্রানজেকশন</h2>
            
            <!-- Button Container -->
            <div class="button-container">
                <button class="filter-btn active" onclick="filterTransactions('all')">
                    <i class="fas fa-list"></i> সব
                </button>
                <button class="filter-btn" onclick="filterTransactions('add')">
                    <i class="fas fa-plus"></i> যোগ
                </button>
                <button class="filter-btn" onclick="filterTransactions('mbank')">
                    <i class="fas fa-mobile-alt"></i> ব্যাংক
                </button>
                <button class="filter-btn" onclick="filterTransactions('other')">
                    <i class="fas fa-ellipsis-h"></i> অন্যান্য
                </button>
            </div>
            
            <!-- History Container -->
            <div class="history-container" id="transactionHistory">
                <?php if (count($transactions) > 0): ?>
                    <?php foreach($transactions as $transaction): ?>
                        <?php 
                            // Determine transaction type and icon
                            $icon = 'fas fa-exchange-alt';
                            $amountClass = '';
                            
                            if ($transaction['type'] == 'add') {
                                $icon = 'fas fa-arrow-down';
                                $amountClass = 'amount-add';
                            } elseif ($transaction['type'] == 'mbank') {
                                $icon = 'fas fa-mobile-alt';
                                $amountClass = 'amount-deduct';
                            } else {
                                $icon = 'fas fa-exchange-alt';
                                $amountClass = 'amount-deduct';
                            }
                            
                            // Determine status class
                            $statusClass = 'status-' . $transaction['status'];
                            
                            // Get transaction info
                            $info = '';
                            if (!empty($transaction['payment_number'])) {
                                $info = htmlspecialchars($transaction['payment_number']);
                            } elseif (!empty($transaction['account_number'])) {
                                $info = htmlspecialchars($transaction['account_number']);
                            } else {
                                $info = "N/A";
                            }
                            
                            // Get reference/note
                            $note = '';
                            if ($transaction['type'] == 'mbank' && !empty($transaction['note'])) {
                                $note = htmlspecialchars($transaction['note']);
                            } elseif (!empty($transaction['reference'])) {
                                $note = htmlspecialchars($transaction['reference']);
                            } else {
                                $note = "N/A";
                            }
                        ?>
                        <div class="history-item" data-type="<?php echo htmlspecialchars($transaction['type']); ?>">
                            <!-- Transaction Header -->
                            <div class="transaction-header">
                                <div class="transaction-title">
                                    <i class="<?php echo $icon; ?>"></i>
                                    <?php echo htmlspecialchars($transaction['title']); ?>
                                </div>
                                <div class="transaction-date">
                                    <?php echo date('d M, Y', strtotime($transaction['date'])); ?>
                                </div>
                            </div>
                            
                            <!-- Transaction Details -->
                            <div class="transaction-details">
                                <div class="detail-item">
                                    <div class="detail-label">তথ্য</div>
                                    <div class="detail-value"><?php echo $info; ?></div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">রেফারেন্স</div>
                                    <div class="detail-value"><?php echo $note; ?></div>
                                </div>
                            </div>
                            
                            <!-- Amount and Status Row -->
                            <div class="amount-status-row">
                                <div class="transaction-amount <?php echo $amountClass; ?>">
                                    <i class="fas fa-money-bill-wave"></i>
                                    ৳<?php echo number_format($transaction['amount'], 2); ?>
                                </div>
                                <div class="status-badge <?php echo $statusClass; ?>">
                                    <i class="fas fa-circle"></i>
                                    <?php echo ucfirst(htmlspecialchars($transaction['status'])); ?>
                                </div>
                            </div>
                            
                            <!-- Note -->
                            <?php if ($note != "N/A"): ?>
                                <div class="transaction-note">
                                    <i class="fas fa-sticky-note"></i> <?php echo $note; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-receipt"></i>
                        <h3>কোন ট্রানজেকশন পাওয়া যায়নি</h3>
                        <p>আপনার এখনো কোন ট্রানজেকশন হিস্টরি নেই।</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Bottom Navigation -->
        <div class="bottom-nav">
            <div class="nav-list">
                <a href="/" class="nav-item">
                    <i class="fas fa-home nav-icon"></i>
                    <div class="nav-label">হোম</div>
                </a>
                <a href="history.php" class="nav-item active">
                    <i class="fas fa-history nav-icon"></i>
                    <div class="nav-label">ইতিহাস</div>
                </a>
                <a href="coin.php" class="nav-item">
                    <i class="fas fa-qrcode nav-icon"></i>
                    <div class="nav-label">স্ক্যান</div>
                </a>
                <a href="coin.php" class="nav-item">
                    <img src="img/tt.png" alt="Rewards" style="width: 16px; height: 14px; margin-bottom: 1px;">
                    <div class="nav-label">রিওয়ার্ড</div>
                </a>
                <a href="profile.php" class="nav-item">
                    <i class="fas fa-user nav-icon"></i>
                    <div class="nav-label">প্রোফাইল</div>
                </a>
            </div>
        </div>
    </div>
    
    <script>
        function filterTransactions(type) {
            const items = document.querySelectorAll('.history-item');
            const buttons = document.querySelectorAll('.filter-btn');
            
            // Update active button
            buttons.forEach(btn => {
                btn.classList.remove('active');
            });
            event.target.classList.add('active');
            
            // Filter transactions
            items.forEach(item => {
                const transactionType = item.getAttribute('data-type');
                if (type === 'all' || transactionType === type) {
                    item.style.display = 'block';
                } else {
                    item.style.display = 'none';
                }
            });
            
            // Check if any transactions are visible
            const visibleItems = Array.from(items).filter(item => item.style.display !== 'none');
            const historyContainer = document.getElementById('transactionHistory');
            
            if (visibleItems.length === 0) {
                if (!document.getElementById('emptyFilterMessage')) {
                    const emptyMessage = document.createElement('div');
                    emptyMessage.id = 'emptyFilterMessage';
                    emptyMessage.className = 'empty-state';
                    emptyMessage.innerHTML = `
                        <i class="fas fa-filter"></i>
                        <h3>কোন ট্রানজেকশন পাওয়া যায়নি</h3>
                        <p>নির্বাচিত ফিল্টারে কোন ট্রানজেকশন নেই।</p>
                    `;
                    historyContainer.appendChild(emptyMessage);
                }
            } else {
                const emptyMessage = document.getElementById('emptyFilterMessage');
                if (emptyMessage) {
                    emptyMessage.remove();
                }
            }
        }
    </script>
</body>
</html>