<?php
session_start();
include('db.php');
// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$userId = $_SESSION['user_id'];
// Fetch all banks from the database
$bankQuery = "SELECT * FROM banks";
$bankResult = $conn->query($bankQuery);
// Fetch user data including balance and PIN
$userQuery = $conn->prepare("SELECT balance, pin FROM users WHERE id = ?");
$userQuery->bind_param("i", $userId);
$userQuery->execute();
$userResult = $userQuery->get_result();
$userData = $userResult->fetch_assoc();
$userBalance = $userData['balance'];
$userPin = $userData['pin'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bank List</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #008080; /* Changed to teal */
            --secondary-color: #4dbdbd; /* Lighter teal for secondary elements */
            --accent-color: #FFD700; /* Kept gold accent */
            --light-bg: #f0f8f8; /* Very light teal background */
            --card-bg: #ffffff;
            --text-primary: #004d40; /* Dark teal for text */
            --text-secondary: #00695c; /* Medium teal for secondary text */
            --shadow: 0 4px 12px rgba(0, 128, 128, 0.15); /* Teal shadow */
            --transition: all 0.3s ease;
        }
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: var(--light-bg);
            color: var(--text-primary);
            line-height: 1.6;
        }
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color)); /* Teal gradient */
            color: white;
            padding: 15px 20px;
            box-shadow: var(--shadow);
            position: sticky;
            top: 0;
            z-index: 100;
        }
        .back-icon {
            display: flex;
            align-items: center;
            cursor: pointer;
            transition: var(--transition);
        }
        .back-icon:hover {
            transform: translateX(-5px);
        }
        .back-icon i {
            font-size: 20px;
            margin-right: 10px;
        }
        .header h3 {
            font-size: 22px;
            font-weight: 600;
            text-align: center;
            flex-grow: 1;
            margin: 0;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.2);
        }
        .balance {
            display: flex;
            align-items: center;
            background-color: rgba(255, 255, 255, 0.2);
            padding: 8px 15px;
            border-radius: 30px;
            font-weight: 600;
            font-size: 16px;
            backdrop-filter: blur(5px);
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .balance i {
            margin-right: 8px;
            color: var(--accent-color);
        }
        .main-container {
            max-width: 800px;
            margin: 25px auto;
            padding: 0 15px;
        }
        .page-title {
            text-align: center;
            margin: 30px 0 20px;
            color: var(--primary-color);
            position: relative;
            font-size: 24px;
            font-weight: 600;
        }
        .page-title::after {
            content: '';
            position: absolute;
            bottom: -10px;
            left: 50%;
            transform: translateX(-50%);
            width: 80px;
            height: 3px;
            background: linear-gradient(to right, var(--primary-color), var(--secondary-color)); /* Teal gradient */
            border-radius: 3px;
        }
        .search-container {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 25px;
            box-shadow: var(--shadow);
            position: relative;
        }
        .search-container i {
            position: absolute;
            left: 20px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-secondary);
            font-size: 18px;
        }
        .search-container input {
            width: 100%;
            padding: 15px 15px 15px 45px;
            font-size: 16px;
            border: 2px solid #b2dfdb; /* Light teal border */
            border-radius: 8px;
            box-sizing: border-box;
            transition: var(--transition);
            background-color: #f0f8f8; /* Very light teal background */
        }
        .search-container input:focus {
            border-color: var(--primary-color);
            background-color: white;
            outline: none;
            box-shadow: 0 0 0 3px rgba(0, 128, 128, 0.1); /* Teal focus shadow */
        }
        .bank-list-container {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        .bank-item {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 20px;
            box-shadow: var(--shadow);
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            border-left: 4px solid transparent;
        }
        .bank-item:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 20px rgba(0, 128, 128, 0.2); /* Teal shadow */
            border-left-color: var(--primary-color);
        }
        .bank-logo {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            object-fit: contain;
            background-color: #f0f8f8; /* Very light teal background */
            padding: 10px;
            margin-right: 15px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .bank-info {
            flex-grow: 1;
        }
        .bank-name {
            font-size: 18px;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 5px;
        }
        .branch-name {
            font-size: 14px;
            color: var(--text-secondary);
            display: flex;
            align-items: center;
        }
        .branch-name i {
            margin-right: 5px;
            color: var(--accent-color);
        }
        .arrow-icon {
            color: var(--text-secondary);
            transition: var(--transition);
        }
        .bank-item:hover .arrow-icon {
            color: var(--primary-color);
            transform: translateX(5px);
        }
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: var(--text-secondary);
            grid-column: 1 / -1;
        }
        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: var(--primary-color);
        }
        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--text-primary);
        }
        @media (max-width: 768px) {
            .header {
                padding: 12px 15px;
            }
            
            .header h3 {
                font-size: 18px;
            }
            
            .balance {
                font-size: 14px;
                padding: 6px 12px;
            }
            
            .page-title {
                font-size: 20px;
            }
            
            .search-container {
                padding: 15px;
            }
            
            .search-container input {
                padding: 12px 12px 12px 40px;
            }
            
            .bank-list-container {
                grid-template-columns: 1fr;
            }
            
            .bank-item {
                padding: 15px;
            }
            
            .bank-logo {
                width: 50px;
                height: 50px;
            }
            
            .bank-name {
                font-size: 16px;
            }
        }
    </style>
</head>
<body>
    <!-- Header Section -->
    <div class="header">
        <div class="back-icon" onclick="history.back()">
            <i class="fas fa-arrow-left"></i>
        </div>
        <h3>Bank List</h3>
        <div class="balance">
            <i class="fas fa-wallet"></i>
            ৳<?php echo number_format($userBalance, 2); ?>
        </div>
    </div>
    <!-- Main Container -->
    <div class="main-container">
        <h2 class="page-title">Select Your Bank</h2>
        
        <!-- Search Container -->
        <div class="search-container">
            <i class="fas fa-search"></i>
            <input type="text" id="searchInput" onkeyup="searchBanks()" placeholder="Search for banks...">
        </div>
        
        <!-- Bank List Container -->
        <div class="bank-list-container" id="bankList">
            <?php if ($bankResult->num_rows > 0): ?>
                <?php while ($row = $bankResult->fetch_assoc()): ?>
                    <div class="bank-item" onclick="selectBank(<?php echo $row['id']; ?>)">
                        <img src="<?php echo htmlspecialchars($row['logo_url']); ?>" alt="<?php echo htmlspecialchars($row['name']); ?>" class="bank-logo">
                        <div class="bank-info">
                            <div class="bank-name"><?php echo htmlspecialchars($row['name']); ?></div>
                            <div class="branch-name">
                                <i class="fas fa-map-marker-alt"></i>
                                <?php echo htmlspecialchars($row['branch']); ?>
                            </div>
                        </div>
                        <i class="fas fa-chevron-right arrow-icon"></i>
                    </div>
                <?php endwhile; ?>
            <?php else: ?>
                <div class="empty-state">
                    <i class="fas fa-university"></i>
                    <h3>No Banks Available</h3>
                    <p>There are currently no banks in our system. Please check back later.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
    <script>
        function searchBanks() {
            // Get the search input and filter value
            let input = document.getElementById('searchInput');
            let filter = input.value.toUpperCase();
            let bankList = document.getElementById('bankList');
            let bankItems = bankList.getElementsByClassName('bank-item');
            let emptyState = bankList.getElementsByClassName('empty-state')[0];
            let visibleCount = 0;
            
            // Loop through all bank items and hide those that don't match the search query
            for (let i = 0; i < bankItems.length; i++) {
                let bankName = bankItems[i].getElementsByClassName('bank-name')[0];
                let branchName = bankItems[i].getElementsByClassName('branch-name')[0];
                let bankText = bankName.textContent || bankName.innerText;
                let branchText = branchName.textContent || branchName.innerText;
                
                if (bankText.toUpperCase().indexOf(filter) > -1 || branchText.toUpperCase().indexOf(filter) > -1) {
                    bankItems[i].style.display = 'flex';
                    visibleCount++;
                } else {
                    bankItems[i].style.display = 'none';
                }
            }
            
            // Show or hide empty state based on search results
            if (visibleCount === 0 && filter !== '') {
                if (!document.getElementById('searchEmptyState')) {
                    let searchEmptyState = document.createElement('div');
                    searchEmptyState.id = 'searchEmptyState';
                    searchEmptyState.className = 'empty-state';
                    searchEmptyState.innerHTML = `
                        <i class="fas fa-search"></i>
                        <h3>No Matching Banks</h3>
                        <p>No banks found matching your search criteria.</p>
                    `;
                    bankList.appendChild(searchEmptyState);
                }
            } else {
                let searchEmptyState = document.getElementById('searchEmptyState');
                if (searchEmptyState) {
                    searchEmptyState.remove();
                }
            }
        }
        
        // Function to handle bank selection and redirect to bank-process.php with bank ID
        function selectBank(bankId) {
            window.location.href = 'bank-process.php?bank_id=' + bankId;
        }
    </script>
</body>
</html>