<?php
session_start();
include('db.php');
// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// Get user data
$userId = $_SESSION['user_id'];
$userQuery = $conn->prepare("SELECT * FROM users WHERE id = ?");
$userQuery->bind_param("i", $userId);
$userQuery->execute();
$userResult = $userQuery->get_result();
$userData = $userResult->fetch_assoc();

// Handle form submissions
$pinError = '';
$passwordError = '';
$successMessage = '';

// Handle PIN change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['pin_change'])) {
    $oldPin = $_POST['old_pin'];
    $newPin = $_POST['new_pin'];
    
    if ($oldPin !== $userData['pin']) {
        $pinError = "বর্তমান PIN সঠিক নয়!";
    } else {
        $updatePinQuery = $conn->prepare("UPDATE users SET pin = ? WHERE id = ?");
        $updatePinQuery->bind_param("si", $newPin, $userId);
        if ($updatePinQuery->execute()) {
            $successMessage = "PIN সফলভাবে আপডেট হয়েছে!";
        } else {
            $pinError = "PIN আপডেট করতে সমস্যা হয়েছে!";
        }
    }
}

// Handle password change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['password_change'])) {
    $oldPassword = $_POST['old_password'];
    $newPassword = $_POST['new_password'];
    
    if (!password_verify($oldPassword, $userData['password'])) {
        $passwordError = "বর্তমান পাসওয়ার্ড সঠিক নয়!";
    } else {
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $updatePasswordQuery = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
        $updatePasswordQuery->bind_param("si", $hashedPassword, $userId);
        if ($updatePasswordQuery->execute()) {
            $successMessage = "পাসওয়ার্ড সফলভাবে আপডেট হয়েছে!";
        } else {
            $passwordError = "পাসওয়ার্ড আপডেট করতে সমস্যা হয়েছে!";
        }
    }
}

// Handle profile image upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['profile_image'])) {
    $file = $_FILES['profile_image'];
    $fileName = $file['name'];
    $fileTmpName = $file['tmp_name'];
    $fileSize = $file['size'];
    $fileError = $file['error'];
    
    if ($fileError === 0) {
        $fileExt = pathinfo($fileName, PATHINFO_EXTENSION);
        $fileNewName = "profile_" . $userId . "." . $fileExt;
        $fileDestination = "uploads/" . $fileNewName;
        
        if (move_uploaded_file($fileTmpName, $fileDestination)) {
            $updateImageQuery = $conn->prepare("UPDATE users SET profile_image = ? WHERE id = ?");
            $updateImageQuery->bind_param("si", $fileNewName, $userId);
            if ($updateImageQuery->execute()) {
                $successMessage = "প্রোফাইল ছবি সফলভাবে আপডেট হয়েছে!";
                // Refresh user data
                $userQuery->execute();
                $userResult = $userQuery->get_result();
                $userData = $userResult->fetch_assoc();
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ইউজার প্রোফাইল</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&family=Noto+Sans+Bengali:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-color: #00897b;
            --primary-light: #b2dfdb;
            --secondary-color: #4db6ac;
            --accent-color: #4db6ac;
            --light-bg: #f5f9f8;
            --card-bg: #ffffff;
            --text-primary: #00695c;
            --text-secondary: #00897b;
            --text-light: #ffffff;
            --shadow: 0 4px 15px rgba(0, 137, 123, 0.08);
            --shadow-hover: 0 8px 25px rgba(0, 137, 123, 0.12);
            --transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
            --radius: 20px;
            --success: #2ecc71;
            --danger: #e74c3c;
            --warning: #f39c12;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Noto Sans Bengali', 'Poppins', sans-serif;
            background-color: var(--light-bg);
            color: var(--text-primary);
            line-height: 1.6;
            padding-bottom: 80px;
            font-weight: 400;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: var(--text-light);
            padding: 18px 22px;
            box-shadow: var(--shadow);
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .back-icon {
            display: flex;
            align-items: center;
            cursor: pointer;
            transition: var(--transition);
            background: rgba(255, 255, 255, 0.15);
            width: 40px;
            height: 40px;
            border-radius: 50%;
            justify-content: center;
        }
        
        .back-icon:hover {
            transform: translateX(-3px);
            background: rgba(255, 255, 255, 0.25);
        }
        
        .back-icon i {
            font-size: 18px;
        }
        
        .header h3 {
            font-size: 20px;
            font-weight: 600;
            text-align: center;
            flex-grow: 1;
            margin: 0;
            text-shadow: 0 1px 2px rgba(0,0,0,0.1);
        }
        
        .balance {
            display: flex;
            align-items: center;
            background-color: rgba(255, 255, 255, 0.2);
            padding: 8px 15px;
            border-radius: 30px;
            font-weight: 600;
            font-size: 15px;
            backdrop-filter: blur(5px);
            box-shadow: 0 2px 6px rgba(0,0,0,0.08);
        }
        
        .balance i {
            margin-right: 6px;
            color: var(--accent-color);
            font-size: 16px;
        }
        
        .main-container {
            max-width: 600px;
            margin: 25px auto;
            padding: 0 15px;
        }
        
        /* Profile Card - Made smaller */
        .profile-card {
            background: var(--card-bg);
            border-radius: var(--radius);
            padding: 20px;
            margin-bottom: 25px;
            box-shadow: var(--shadow);
            position: relative;
            overflow: hidden;
            text-align: center;
        }
        
        .profile-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 4px;
            background: linear-gradient(to right, var(--primary-color), var(--secondary-color));
        }
        
        .profile-image-container {
            position: relative;
            width: 80px;
            height: 80px;
            margin: 0 auto 15px;
        }
        
        .profile-image {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
            border: 3px solid var(--primary-light);
            box-shadow: 0 4px 12px rgba(0, 137, 123, 0.15);
        }
        
        .camera-overlay {
            position: absolute;
            bottom: 0;
            right: 0;
            width: 28px;
            height: 28px;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            cursor: pointer;
            box-shadow: 0 2px 6px rgba(0, 137, 123, 0.3);
            transition: var(--transition);
            border: 2px solid white;
        }
        
        .camera-overlay:hover {
            transform: scale(1.1);
        }
        
        .profile-name {
            font-size: 18px;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 5px;
        }
        
        .profile-phone {
            font-size: 14px;
            color: var(--text-secondary);
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .profile-phone i {
            margin-right: 6px;
            color: var(--accent-color);
            font-size: 12px;
        }
        
        .profile-stats {
            display: flex;
            justify-content: space-around;
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px solid rgba(0, 137, 123, 0.1);
        }
        
        .stat-item {
            text-align: center;
        }
        
        .stat-value {
            font-size: 16px;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 3px;
        }
        
        .stat-label {
            font-size: 12px;
            color: var(--text-secondary);
        }
        
        /* Settings Section */
        .settings-section {
            background: var(--card-bg);
            border-radius: var(--radius);
            padding: 25px;
            margin-bottom: 25px;
            box-shadow: var(--shadow);
        }
        
        .section-title {
            font-size: 18px;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 20px;
            display: flex;
            align-items: center;
        }
        
        .section-title i {
            margin-right: 10px;
            color: var(--secondary-color);
            font-size: 20px;
        }
        
        .settings-list {
            list-style: none;
        }
        
        .setting-item {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 14px 0;
            border-bottom: 1px solid rgba(0, 137, 123, 0.08);
            cursor: pointer;
            transition: var(--transition);
        }
        
        .setting-item:last-child {
            border-bottom: none;
        }
        
        .setting-item:hover {
            padding-left: 5px;
        }
        
        .setting-left {
            display: flex;
            align-items: center;
        }
        
        .setting-icon {
            width: 38px;
            height: 38px;
            border-radius: 10px;
            background: linear-gradient(135deg, var(--primary-light), #e0f2f1);
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 12px;
            color: var(--primary-color);
            font-size: 16px;
            box-shadow: 0 3px 6px rgba(0, 137, 123, 0.1);
        }
        
        .setting-info {
            flex-grow: 1;
        }
        
        .setting-title {
            font-size: 15px;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 2px;
        }
        
        .setting-desc {
            font-size: 12px;
            color: var(--text-secondary);
        }
        
        .setting-arrow {
            color: var(--text-secondary);
            transition: var(--transition);
        }
        
        .setting-item:hover .setting-arrow {
            color: var(--primary-color);
            transform: translateX(3px);
        }
        
        .logout-item .setting-icon {
            background: linear-gradient(135deg, #ffebee, #ffcdd2);
            color: var(--danger);
        }
        
        .logout-item .setting-title {
            color: var(--danger);
        }
        
        /* Toggle Switch */
        .switch {
            position: relative;
            display: inline-block;
            width: 44px;
            height: 22px;
        }
        
        .switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }
        
        .slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            transition: var(--transition);
            border-radius: 22px;
        }
        
        .slider:before {
            position: absolute;
            content: "";
            height: 16px;
            width: 16px;
            left: 3px;
            bottom: 3px;
            background-color: white;
            transition: var(--transition);
            border-radius: 50%;
        }
        
        input:checked + .slider {
            background-color: var(--primary-color);
        }
        
        input:checked + .slider:before {
            transform: translateX(22px);
        }
        
        /* Action Buttons */
        .action-buttons {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin-bottom: 25px;
        }
        
        .action-btn {
            background: var(--card-bg);
            border-radius: var(--radius);
            padding: 15px;
            box-shadow: var(--shadow);
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            text-decoration: none;
            color: var(--text-primary);
            transition: var(--transition);
            position: relative;
            overflow: hidden;
        }
        
        .action-btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 4px;
            background: linear-gradient(to right, var(--primary-color), var(--secondary-color));
            transform: scaleX(0);
            transform-origin: left;
            transition: transform 0.3s ease;
        }
        
        .action-btn:hover::before {
            transform: scaleX(1);
        }
        
        .action-btn:hover {
            transform: translateY(-5px);
            box-shadow: var(--shadow-hover);
        }
        
        .action-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary-light), #e0f2f1);
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 8px;
            color: var(--primary-color);
            font-size: 18px;
            box-shadow: 0 3px 6px rgba(0, 137, 123, 0.1);
        }
        
        .action-title {
            font-size: 14px;
            font-weight: 600;
            color: var(--primary-color);
        }
        
        /* Popup Overlay */
        .popup-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .popup-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .popup {
            background: var(--card-bg);
            border-radius: var(--radius);
            width: 90%;
            max-width: 400px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.2);
            transform: scale(0.8);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .popup-overlay.active .popup {
            transform: scale(1);
        }
        
        .popup::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 5px;
            background: linear-gradient(to right, var(--primary-color), var(--secondary-color));
        }
        
        .popup-header {
            padding: 25px 25px 15px;
            text-align: center;
        }
        
        .popup-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .popup-title i {
            margin-right: 10px;
            color: var(--secondary-color);
        }
        
        .popup-body {
            padding: 0 25px 25px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group input {
            width: 100%;
            padding: 15px;
            border: 2px solid #e0f2f1;
            border-radius: var(--radius);
            font-size: 15px;
            transition: var(--transition);
            background-color: var(--light-bg);
            font-weight: 500;
        }
        
        .form-group input:focus {
            border-color: var(--primary-color);
            background-color: var(--card-bg);
            outline: none;
            box-shadow: 0 0 0 3px rgba(0, 137, 123, 0.08);
        }
        
        .popup-buttons {
            display: flex;
            gap: 12px;
        }
        
        .popup-btn {
            flex: 1;
            padding: 14px;
            border: none;
            border-radius: var(--radius);
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
        }
        
        .submit-btn {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            box-shadow: 0 4px 8px rgba(0, 137, 123, 0.2);
        }
        
        .submit-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 12px rgba(0, 137, 123, 0.3);
        }
        
        .cancel-btn {
            background: #f5f5f5;
            color: var(--text-secondary);
        }
        
        .cancel-btn:hover {
            background: #e0e0e0;
        }
        
        /* Toast Notification */
        .toast {
            position: fixed;
            top: 20px;
            left: 50%;
            transform: translateX(-50%) translateY(-100px);
            background: white;
            color: var(--text-primary);
            padding: 15px 25px;
            border-radius: var(--radius);
            box-shadow: var(--shadow-hover);
            z-index: 2000;
            display: flex;
            align-items: center;
            opacity: 0;
            transition: all 0.4s ease;
            min-width: 300px;
            max-width: 90%;
        }
        
        .toast.show {
            transform: translateX(-50%) translateY(0);
            opacity: 1;
        }
        
        .toast i {
            font-size: 24px;
            margin-right: 15px;
        }
        
        .toast.success {
            border-left: 5px solid var(--success);
        }
        
        .toast.success i {
            color: var(--success);
        }
        
        .toast.error {
            border-left: 5px solid var(--danger);
        }
        
        .toast.error i {
            color: var(--danger);
        }
        
        /* Bottom Navigation */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            box-shadow: 0 -3px 12px rgba(0, 0, 0, 0.06);
            z-index: 100;
            border-top: 1px solid rgba(0, 137, 123, 0.08);
        }
        
        .nav-list {
            display: flex;
            justify-content: space-around;
            padding: 10px 0;
        }
        
        .nav-item {
            text-align: center;
            text-decoration: none;
            color: var(--text-secondary);
            transition: var(--transition);
            padding: 5px 8px;
            border-radius: 12px;
            min-width: 60px;
            position: relative;
        }
        
        .nav-item.active {
            color: var(--primary-color);
        }
        
        .nav-item.active::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 50%;
            transform: translateX(-50%);
            width: 24px;
            height: 3px;
            background: var(--primary-color);
            border-radius: 3px;
        }
        
        .nav-icon {
            font-size: 18px;
            margin-bottom: 3px;
        }
        
        .nav-label {
            font-size: 10px;
            font-weight: 500;
        }
        
        /* Responsive Design */
        @media (max-width: 768px) {
            .header {
                padding: 14px 18px;
            }
            
            .header h3 {
                font-size: 18px;
            }
            
            .balance {
                font-size: 13px;
                padding: 6px 12px;
            }
            
            .profile-card {
                padding: 18px 15px;
            }
            
            .profile-image-container {
                width: 70px;
                height: 70px;
            }
            
            .profile-name {
                font-size: 16px;
            }
            
            .settings-section {
                padding: 20px;
            }
            
            .action-buttons {
                grid-template-columns: 1fr;
            }
            
            .popup {
                width: 95%;
                max-width: 350px;
            }
        }
    </style>
</head>
<body>
    <!-- Header Section -->
    <div class="header">
        <div class="back-icon" onclick="history.back()">
            <i class="fas fa-arrow-left"></i>
        </div>
        <h3>ইউজার প্রোফাইল</h3>
        <div class="balance">
            <i class="fas fa-wallet"></i>
            ৳<?php echo number_format($userData['balance'], 2); ?>
        </div>
    </div>
    
    <!-- Main Container -->
    <div class="main-container">
        <!-- Profile Card - Made smaller -->
        <div class="profile-card">
            <div class="profile-image-container">
                <?php 
                    // Use profile image if exists, otherwise use default
                    $profileImage = !empty($userData['profile_image']) ? 'uploads/' . $userData['profile_image'] : 'image/logo.jpg';
                ?>
                <img src="<?php echo $profileImage; ?>" alt="User Profile" class="profile-image">
                <div class="camera-overlay" onclick="document.getElementById('profileImageInput').click()">
                    <i class="fas fa-camera"></i>
                </div>
            </div>
            
            <div class="profile-name"><?php echo htmlspecialchars($userData['name']); ?></div>
            <div class="profile-phone">
                <i class="fas fa-phone-alt"></i>
                <?php echo htmlspecialchars($userData['phone']); ?>
            </div>
            
            <div class="profile-stats">
                <div class="stat-item">
                    <div class="stat-value"><?php echo number_format($userData['balance'], 0); ?></div>
                    <div class="stat-label">ব্যালেন্স</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value"><?php echo number_format($userData['reward_points'], 0); ?></div>
                    <div class="stat-label">রিওয়ার্ড পয়েন্ট</div>
                </div>
            </div>
        </div>
        
        <!-- Hidden file input for uploading profile image -->
        <form method="POST" enctype="multipart/form-data">
            <input type="file" name="profile_image" id="profileImageInput" style="display:none;" onchange="this.form.submit();" accept="image/*">
        </form>
        
        <!-- Settings Section -->
        <div class="settings-section">
            <div class="section-title">
                <i class="fas fa-cog"></i>
                অ্যাকাউন্ট সেটিংস
            </div>
            
            <ul class="settings-list">
                <li class="setting-item" onclick="showPopup('pinPopup')">
                    <div class="setting-left">
                        <div class="setting-icon">
                            <i class="fas fa-key"></i>
                        </div>
                        <div class="setting-info">
                            <div class="setting-title">PIN পরিবর্তন</div>
                            <div class="setting-desc">আপনার সুরক্ষা PIN আপডেট করুন</div>
                        </div>
                    </div>
                    <i class="fas fa-chevron-right setting-arrow"></i>
                </li>
                
                <li class="setting-item" onclick="showPopup('passwordPopup')">
                    <div class="setting-left">
                        <div class="setting-icon">
                            <i class="fas fa-lock"></i>
                        </div>
                        <div class="setting-info">
                            <div class="setting-title">পাসওয়ার্ড পরিবর্তন</div>
                            <div class="setting-desc">আপনার অ্যাকাউন্ট পাসওয়ার্ড পরিবর্তন করুন</div>
                        </div>
                    </div>
                    <i class="fas fa-chevron-right setting-arrow"></i>
                </li>
                
                <li class="setting-item">
                    <div class="setting-left">
                        <div class="setting-icon">
                            <i class="fas fa-bell"></i>
                        </div>
                        <div class="setting-info">
                            <div class="setting-title">নোটিফিকেশন</div>
                            <div class="setting-desc">আপনার সতর্কতা পছন্দগুলি পরিচালনা করুন</div>
                        </div>
                    </div>
                    <label class="switch">
                        <input type="checkbox" checked>
                        <span class="slider"></span>
                    </label>
                </li>
                
                <li class="setting-item" onclick="window.location.href='reward_convert.php'">
                    <div class="setting-left">
                        <div class="setting-icon">
                            <i class="fas fa-gift"></i>
                        </div>
                        <div class="setting-info">
                            <div class="setting-title">রিওয়ার্ড</div>
                            <div class="setting-desc">রিওয়ার্ড কয়েন ব্যালেন্সে রূপান্তর করুন</div>
                        </div>
                    </div>
                    <i class="fas fa-chevron-right setting-arrow"></i>
                </li>
                
                <li class="setting-item logout-item" onclick="window.location.href='logout.php'">
                    <div class="setting-left">
                        <div class="setting-icon">
                            <i class="fas fa-sign-out-alt"></i>
                        </div>
                        <div class="setting-info">
                            <div class="setting-title">লগ আউট</div>
                            <div class="setting-desc">আপনার অ্যাকাউন্ট থেকে সাইন আউট করুন</div>
                        </div>
                    </div>
                    <i class="fas fa-chevron-right setting-arrow"></i>
                </li>
            </ul>
        </div>
        
        <!-- Action Buttons -->
        <div class="action-buttons">
            <a href="/App" class="action-btn">
                <div class="action-icon">
                    <i class="fas fa-download"></i>
                </div>
                <div class="action-title">অ্যাপ ডাউনলোড</div>
            </a>
            <a href="https://wa.me/+8801796003076" class="action-btn">
                <div class="action-icon">
                    <i class="fas fa-comments"></i>
                </div>
                <div class="action-title">যোগাযোগ করুন</div>
            </a>
        </div>
    </div>
    
    <!-- PIN Change Popup -->
    <div class="popup-overlay" id="pinPopup">
        <div class="popup">
            <div class="popup-header">
                <div class="popup-title">
                    <i class="fas fa-key"></i> PIN পরিবর্তন
                </div>
            </div>
            <div class="popup-body">
                <form method="POST">
                    <input type="hidden" name="pin_change" value="1">
                    <div class="form-group">
                        <input type="password" name="old_pin" placeholder="বর্তমান PIN" required>
                    </div>
                    <div class="form-group">
                        <input type="password" name="new_pin" placeholder="নতুন PIN" required>
                    </div>
                    <div class="popup-buttons">
                        <button type="submit" class="popup-btn submit-btn">আপডেট করুন</button>
                        <button type="button" class="popup-btn cancel-btn" onclick="closePopup('pinPopup')">বাতিল করুন</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Password Change Popup -->
    <div class="popup-overlay" id="passwordPopup">
        <div class="popup">
            <div class="popup-header">
                <div class="popup-title">
                    <i class="fas fa-lock"></i> পাসওয়ার্ড পরিবর্তন
                </div>
            </div>
            <div class="popup-body">
                <form method="POST">
                    <input type="hidden" name="password_change" value="1">
                    <div class="form-group">
                        <input type="password" name="old_password" placeholder="বর্তমান পাসওয়ার্ড" required>
                    </div>
                    <div class="form-group">
                        <input type="password" name="new_password" placeholder="নতুন পাসওয়ার্ড" required>
                    </div>
                    <div class="popup-buttons">
                        <button type="submit" class="popup-btn submit-btn">আপডেট করুন</button>
                        <button type="button" class="popup-btn cancel-btn" onclick="closePopup('passwordPopup')">বাতিল করুন</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Toast Notification -->
    <div id="toast" class="toast <?php echo ($pinError || $passwordError) ? 'error' : 'success'; ?>">
        <i class="fas <?php echo ($pinError || $passwordError) ? 'fa-exclamation-circle' : 'fa-check-circle'; ?>"></i>
        <span><?php echo $pinError ? $pinError : ($passwordError ? $passwordError : $successMessage); ?></span>
    </div>
    
    <!-- Bottom Navigation -->
    <div class="bottom-nav">
        <div class="nav-list">
            <a href="/" class="nav-item">
                <i class="fas fa-home nav-icon"></i>
                <div class="nav-label">হোম</div>
            </a>
            <a href="history.php" class="nav-item">
                <i class="fas fa-history nav-icon"></i>
                <div class="nav-label">ইতিহাস</div>
            </a>
            <a href="coin.php" class="nav-item">
                <i class="fas fa-qrcode nav-icon"></i>
                <div class="nav-label">স্ক্যান</div>
            </a>
            <a href="coin.php" class="nav-item">
                <img src="img/tt.png" alt="রিওয়ার্ড" style="width: 16px; height: 14px; margin-bottom: 1px;">
                <div class="nav-label">রিওয়ার্ড</div>
            </a>
            <a href="profile.php" class="nav-item active">
                <i class="fas fa-user nav-icon"></i>
                <div class="nav-label">প্রোফাইল</div>
            </a>
        </div>
    </div>
    
    <script>
        function showPopup(popupId) {
            const popup = document.getElementById(popupId);
            popup.classList.add('active');
        }
        
        function closePopup(popupId) {
            const popup = document.getElementById(popupId);
            popup.classList.remove('active');
        }
        
        // Close popup when clicking outside
        document.querySelectorAll('.popup-overlay').forEach(popup => {
            popup.addEventListener('click', function(e) {
                if (e.target === this) {
                    this.classList.remove('active');
                }
            });
        });
        
        // Show toast notification if there's a message
        <?php if ($successMessage || $pinError || $passwordError): ?>
            const toast = document.getElementById('toast');
            setTimeout(() => {
                toast.classList.add('show');
            }, 100);
            
            setTimeout(() => {
                toast.classList.remove('show');
            }, 3000);
        <?php endif; ?>
    </script>
</body>
</html>