<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
include('db.php');
// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$userId = $_SESSION['user_id'];
$loanError = '';
$successMessage = '';
$showErrorToast = false;
// Fetch user data from dashboard
$userQuery = $conn->prepare("SELECT name, balance, reward_points, profile_image, pin FROM users WHERE id = ?");
$userQuery->bind_param("i", $userId);
$userQuery->execute();
$userResult = $userQuery->get_result();
$userData = $userResult->fetch_assoc();
$userBalance = $userData['balance'];
$userName = $userData['name'];
$rewardPoints = $userData['reward_points'];
$minRequiredBalance = 5000;
$hasEnoughBalance = $userBalance >= $minRequiredBalance;
// Fetch loan settings from database
$settingsQuery = $conn->prepare("SELECT loan_notice, min_loan_amount, max_loan_amount, interest_rate FROM settings WHERE id = 1");
$settingsQuery->execute();
$settingsResult = $settingsQuery->get_result();
$settingsData = $settingsResult->fetch_assoc();
$loanNotice = $settingsData['loan_notice'] ?? 'লোনের জন্য আবেদন করুন। সর্বোচ্চ ৳৫০,০০০ পর্যন্ত লোন নেওয়া যাবে।';
$minLoanAmount = $settingsData['min_loan_amount'] ?? 1000;
$maxLoanAmount = $settingsData['max_loan_amount'] ?? 50000;
$interestRate = $settingsData['interest_rate'] ?? 5.00;
// Fetch user's existing loans
$loansQuery = $conn->prepare("SELECT id, amount, purpose, term_months, status, created_at FROM loans WHERE user_id = ? ORDER BY created_at DESC");
$loansQuery->bind_param("i", $userId);
$loansQuery->execute();
$loansResult = $loansQuery->get_result();
$existingLoans = $loansResult->fetch_all(MYSQLI_ASSOC);
// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check if user has enough balance
    if (!$hasEnoughBalance) {
        $loanError = "You need at least ৳$minRequiredBalance in your account to apply for a loan. Your current balance is ৳$userBalance.";
        $showErrorToast = true;
    } else {
        $loanAmount = $_POST['loan_amount'];
        $loanPurpose = $_POST['loan_purpose'];
        $loanTerm = $_POST['loan_term'];
        $loanDate = $_POST['loan_date'];
        $inputPin = $_POST['pin'];
        
        // Validate PIN
        if ($inputPin !== $userData['pin']) {
            $loanError = "Invalid PIN!";
            $showErrorToast = true;
        } 
        // Validate loan amount
        elseif ($loanAmount < $minLoanAmount || $loanAmount > $maxLoanAmount) {
            $loanError = "Loan amount must be between ৳$minLoanAmount and ৳$maxLoanAmount";
            $showErrorToast = true;
        }
        // Validate loan date
        elseif (empty($loanDate)) {
            $loanError = "Please select a loan date";
            $showErrorToast = true;
        }
        else {
            // Calculate interest and total repayment
            $interestAmount = ($loanAmount * $interestRate) / 100;
            $totalRepayment = $loanAmount + $interestAmount;
            
            // Insert loan application into database
            $insertLoanQuery = $conn->prepare("INSERT INTO loans (user_id, amount, purpose, term_months, interest_rate, total_repayment, status, loan_date) VALUES (?, ?, ?, ?, ?, ?, 'Pending', ?)");
            $insertLoanQuery->bind_param("isdsdds", $userId, $loanAmount, $loanPurpose, $loanTerm, $interestRate, $totalRepayment, $loanDate);
            
            if ($insertLoanQuery->execute()) {
                $successMessage = "Loan application submitted successfully!";
                
                // Send notification
                function sendOneSignalNotificationToUser($title, $message, $userId, $url = null, $imageUrl = null) {
                    $content = array(
                        "en" => $message
                    );
                    $fields = array(
                        'app_id' => "99424ad8-e933-4396-bc62-40c08c6d5b8d",
                        'include_player_ids' => array($userId),
                        'headings' => array("en" => $title),
                        'contents' => $content
                    );
                    
                    if ($url) {
                        $fields['url'] = $url;
                    }
                    
                    if ($imageUrl) {
                        $fields['big_picture'] = $imageUrl;
                    }
                    
                    $fields = json_encode($fields);
                    $ch = curl_init();
                    curl_setopt($ch, CURLOPT_URL, "https://onesignal.com/api/v1/notifications");
                    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                        'Content-Type: application/json; charset=utf-8',
                        'Authorization: Basic NjdhMzQ5OTktNWU1NC00MDBjLWE1YjktMTBhZmU4YzNlZDJl'
                    ));
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
                    curl_setopt($ch, CURLOPT_HEADER, FALSE);
                    curl_setopt($ch, CURLOPT_POST, TRUE);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $fields);
                    curl_setopt($ch, CURLOPT_SSL_VERIFPEER, FALSE);
                    $response = curl_exec($ch);
                    curl_close($ch);
                    return $response;
                }
                
                $title = "Loan Application";
                $message = "Your loan application of ৳$loanAmount has been submitted successfully.";
                sendOneSignalNotificationToUser($title, $message, $userId);
                
            } else {
                $loanError = "Error submitting loan application: " . $conn->error;
                $showErrorToast = true;
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>লোন - Ramit Wallet</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Hind+Siliguri:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Hind Siliguri', sans-serif;
            -webkit-tap-highlight-color: transparent;
        }
        
        :root {
            --teal-primary: #008080;
            --teal-light: #e0fffe;
            --teal-medium: #4dbdbd;
            --teal-dark: #004d40;
            --teal-bg-light: #f0f8f8;
            --teal-text-dark: #004d40;
            --teal-text-medium: #00695c;
            --teal-text-light: #00a39a;
            --teal-gray-lightest: #f5fafa;
            --teal-gray-light: #e0f2f1;
            --teal-gray: #80cbc4;
            --teal-gray-dark: #00695c;
            --teal-border: #b2dfdb;
            --teal-success: #2ecc71;
            --teal-warning: #f39c12;
            --teal-danger: #e74c3c;
            --teal-blue: #3498db;
            --transition: all 0.3s ease;
        }
        
        body {
            background: var(--teal-bg-light);
            color: var(--teal-text-dark);
            font-size: 14px;
            line-height: 1.4;
        }
        
        .app-container {
            max-width: 480px;
            margin: 0 auto;
            min-height: 100vh;
            background: #ffffff;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
            padding-bottom: 70px;
        }
        
        /* Header Styles */
        .app-header {
            background: white;
            padding: 15px;
            color: var(--teal-text-dark);
            position: relative;
            border-bottom: 1px solid var(--teal-border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .back-btn {
            background: var(--teal-light);
            border: none;
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--teal-primary);
            cursor: pointer;
            transition: var(--transition);
        }
        
        .back-btn:hover {
            background: var(--teal-primary);
            color: white;
            transform: scale(1.05);
        }
        
        .header-title {
            font-size: 18px;
            font-weight: 600;
            text-align: center;
            flex-grow: 1;
        }
        
        .balance-badge {
            background: var(--teal-light);
            border-radius: 20px;
            padding: 6px 12px;
            font-size: 14px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 5px;
            color: var(--teal-primary);
        }
        
        /* Balance Requirement */
        .balance-requirement {
            margin: 15px;
            background: <?php echo $hasEnoughBalance ? 'var(--teal-light)' : '#fff5f5'; ?>;
            border-radius: 8px;
            padding: 12px;
            border-left: 4px solid <?php echo $hasEnoughBalance ? 'var(--teal-primary)' : 'var(--teal-danger)'; ?>;
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
        }
        
        .balance-requirement-text {
            color: <?php echo $hasEnoughBalance ? 'var(--teal-primary)' : 'var(--teal-danger)'; ?>;
            font-size: 13px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .balance-progress {
            margin-top: 10px;
            height: 8px;
            background: var(--teal-gray-light);
            border-radius: 4px;
            overflow: hidden;
        }
        
        .balance-progress-bar {
            height: 100%;
            background: <?php echo $hasEnoughBalance ? 'var(--teal-success)' : 'var(--teal-warning)'; ?>;
            width: <?php echo min(100, ($userBalance / $minRequiredBalance) * 100); ?>%;
            transition: width 1s ease;
        }
        
        .balance-stats {
            display: flex;
            justify-content: space-between;
            margin-top: 5px;
            font-size: 12px;
        }
        
        .balance-current {
            color: var(--teal-text-medium);
        }
        
        .balance-required {
            color: var(--teal-text-medium);
        }
        
        /* Notice Container */
        .notice-container {
            margin: 15px;
            background: var(--teal-light);
            border-radius: 8px;
            padding: 12px;
            border-left: 4px solid var(--teal-primary);
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .notice-text {
            color: var(--teal-primary);
            font-size: 13px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .close-notice {
            background: none;
            border: none;
            color: var(--teal-primary);
            font-size: 16px;
            cursor: pointer;
            transition: var(--transition);
        }
        
        .close-notice:hover {
            transform: scale(1.2);
        }
        
        /* Balance Card */
        .balance-card {
            margin: 15px;
            background: linear-gradient(135deg, var(--teal-primary), var(--teal-dark));
            border-radius: 15px;
            padding: 20px;
            color: white;
            box-shadow: 0 4px 12px rgba(0, 128, 128, 0.15);
            position: relative;
            overflow: hidden;
        }
        
        .balance-greeting {
            font-size: 16px;
            opacity: 0.9;
            margin-bottom: 5px;
            display: flex;
            align-items: center;
        }
        
        .balance-greeting i {
            margin-right: 8px;
            font-size: 18px;
        }
        
        .balance-label { 
            font-size: 14px; 
            opacity: 0.9; 
            margin-bottom: 5px; 
        }
        
        .balance-amount { 
            font-size: 28px; 
            font-weight: 700; 
            margin-bottom: 15px; 
        }
        
        .reward-points {
            display: flex;
            align-items: center;
            gap: 5px;
            font-size: 14px;
            opacity: 0.9;
        }
        
        /* Form Container */
        .form-container {
            padding: 15px;
        }
        
        .form-title {
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 15px;
            color: var(--teal-text-dark);
            text-align: center;
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 6px;
            font-weight: 500;
            color: var(--teal-gray);
            font-size: 13px;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid var(--teal-border);
            border-radius: 8px;
            font-size: 14px;
            transition: var(--transition);
            background-color: white;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--teal-primary);
            box-shadow: 0 0 0 2px rgba(0, 128, 128, 0.2);
        }
        
        .form-control:disabled {
            background-color: var(--teal-gray-lightest);
            cursor: not-allowed;
            opacity: 0.7;
        }
        
        /* Custom Select Dropdown */
        .select-wrapper {
            position: relative;
        }
        
        .select-wrapper::after {
            content: '\f107';
            font-family: 'Font Awesome 6 Free';
            font-weight: 900;
            position: absolute;
            top: 50%;
            right: 15px;
            transform: translateY(-50%);
            pointer-events: none;
            color: var(--teal-gray);
        }
        
        select.form-control {
            appearance: none;
            padding-right: 40px;
            cursor: pointer;
        }
        
        select.form-control:disabled {
            cursor: not-allowed;
        }
        
        /* Input with Icon */
        .input-with-icon {
            position: relative;
        }
        
        .input-icon {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--teal-gray);
        }
        
        .input-with-icon .form-control {
            padding-left: 40px;
        }
        
        /* Submit Button */
        .submit-btn {
            background: var(--teal-primary);
            color: white;
            border: none;
            padding: 12px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            width: 100%;
            transition: var(--transition);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .submit-btn:hover:not(:disabled) {
            background: var(--teal-dark);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 128, 128, 0.3);
        }
        
        .submit-btn:active:not(:disabled) {
            transform: translateY(0);
        }
        
        .submit-btn:disabled {
            background: var(--teal-gray);
            cursor: not-allowed;
            opacity: 0.7;
        }
        
        /* Loan Offers */
        .loan-offers {
            padding: 0 15px 15px;
        }
        
        .section-title {
            font-size: 18px;
            font-weight: 600;
            color: var(--teal-primary);
            margin-bottom: 15px;
            text-align: center;
            position: relative;
        }
        
        .section-title::after {
            content: '';
            position: absolute;
            bottom: -6px;
            left: 50%;
            transform: translateX(-50%);
            width: 50px;
            height: 3px;
            background: var(--teal-primary);
            border-radius: 3px;
        }
        
        .offers-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
        }
        
        .offer-card {
            background: white;
            border-radius: 12px;
            padding: 15px;
            box-shadow: 0 3px 8px rgba(0, 0, 0, 0.08);
            transition: var(--transition);
            border: 1px solid var(--teal-border);
            cursor: pointer;
            opacity: <?php echo $hasEnoughBalance ? '1' : '0.6'; ?>;
            pointer-events: <?php echo $hasEnoughBalance ? 'auto' : 'none'; ?>;
        }
        
        .offer-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 12px rgba(0, 128, 128, 0.15);
        }
        
        .offer-card.selected {
            border-color: var(--teal-primary);
            background: var(--teal-light);
        }
        
        .offer-title {
            font-size: 16px;
            font-weight: 600;
            color: var(--teal-primary);
            margin-bottom: 8px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .offer-details {
            font-size: 13px;
            color: var(--teal-text-medium);
            margin-bottom: 5px;
        }
        
        .offer-amount {
            font-size: 18px;
            font-weight: 700;
            color: var(--teal-text-dark);
            margin-top: 8px;
        }
        
        /* Existing Loans */
        .existing-loans {
            padding: 0 15px 15px;
        }
        
        .loan-list {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }
        
        .loan-item {
            background: white;
            border-radius: 12px;
            padding: 15px;
            box-shadow: 0 3px 8px rgba(0, 0, 0, 0.08);
            border-left: 4px solid var(--teal-primary);
        }
        
        .loan-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 8px;
        }
        
        .loan-id {
            font-size: 13px;
            color: var(--teal-text-medium);
        }
        
        .loan-status {
            font-size: 12px;
            font-weight: 600;
            padding: 4px 8px;
            border-radius: 12px;
        }
        
        .status-pending {
            background: var(--teal-warning);
            color: white;
        }
        
        .status-approved {
            background: var(--teal-success);
            color: white;
        }
        
        .status-rejected {
            background: var(--teal-danger);
            color: white;
        }
        
        .loan-amount {
            font-size: 18px;
            font-weight: 700;
            color: var(--teal-text-dark);
            margin-bottom: 5px;
        }
        
        .loan-purpose {
            font-size: 14px;
            color: var(--teal-text-medium);
            margin-bottom: 5px;
        }
        
        .loan-term {
            font-size: 13px;
            color: var(--teal-gray);
        }
        
        /* Popup Styles */
        .popup-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            animation: fadeIn 0.3s ease;
        }
        
        .popup-overlay.show {
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .popup {
            background: white;
            border-radius: 12px;
            width: 90%;
            max-width: 320px;
            padding: 20px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
            animation: slideUp 0.3s ease;
        }
        
        @keyframes slideUp {
            from { 
                opacity: 0;
                transform: translateY(30px);
            }
            to { 
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .popup-header {
            text-align: center;
            margin-bottom: 15px;
        }
        
        .popup-title {
            font-size: 18px;
            font-weight: 600;
            color: var(--teal-text-dark);
        }
        
        .popup-body {
            margin-bottom: 15px;
        }
        
        .pin-input {
            width: 100%;
            padding: 12px;
            border: 1px solid var(--teal-border);
            border-radius: 8px;
            font-size: 16px;
            text-align: center;
            letter-spacing: 5px;
            transition: var(--transition);
        }
        
        .pin-input:focus {
            outline: none;
            border-color: var(--teal-primary);
            box-shadow: 0 0 0 2px rgba(0, 128, 128, 0.2);
        }
        
        .popup-footer {
            display: flex;
            gap: 10px;
        }
        
        .popup-btn {
            flex: 1;
            padding: 10px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
        }
        
        .verify-btn {
            background: var(--teal-primary);
            color: white;
        }
        
        .verify-btn:hover {
            background: var(--teal-dark);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 128, 128, 0.3);
        }
        
        .cancel-btn {
            background: var(--teal-gray-lightest);
            color: var(--teal-gray);
        }
        
        .cancel-btn:hover {
            background: #e0e0e0;
        }
        
        /* Toast Notification */
        .toast {
            visibility: hidden;
            position: fixed;
            left: 50%;
            bottom: 30px;
            transform: translateX(-50%);
            background: white;
            color: var(--teal-text-dark);
            padding: 12px 16px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            z-index: 2000;
            display: flex;
            align-items: center;
            gap: 8px;
            min-width: 250px;
            opacity: 0;
            transition: opacity 0.3s, bottom 0.3s;
        }
        
        .toast.show {
            visibility: visible;
            opacity: 1;
            bottom: 50px;
        }
        
        .toast.success {
            border-left: 4px solid var(--teal-success);
        }
        
        .toast.error {
            border-left: 4px solid var(--teal-danger);
        }
        
        .toast-icon {
            font-size: 18px;
        }
        
        .toast.success .toast-icon {
            color: var(--teal-success);
        }
        
        .toast.error .toast-icon {
            color: var(--teal-danger);
        }
        
        .toast-message {
            font-weight: 500;
        }
        
        /* Form Card */
        .form-card {
            background: white;
            border-radius: 12px;
            margin: 15px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            overflow: hidden;
        }
        
        .form-card-header {
            background: var(--teal-primary);
            color: white;
            padding: 12px;
            text-align: center;
        }
        
        .form-card-body {
            padding: 15px;
        }
        
        /* Loan Type Selection */
        .loan-type-selection {
            margin-bottom: 15px;
        }
        
        .loan-type-details {
            background: var(--teal-light);
            border-radius: 8px;
            padding: 12px;
            margin-top: 10px;
            display: none;
        }
        
        .loan-type-details.active {
            display: block;
        }
        
        .loan-type-detail-item {
            display: flex;
            justify-content: space-between;
            margin-bottom: 5px;
            font-size: 13px;
        }
        
        .loan-type-detail-label {
            color: var(--teal-text-medium);
        }
        
        .loan-type-detail-value {
            font-weight: 600;
            color: var(--teal-primary);
        }
        
        /* Bottom Navigation */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            box-shadow: 0 -2px 8px rgba(0, 0, 0, 0.1);
            z-index: 100;
            border-top: 1px solid var(--teal-border);
        }
        
        .nav-list {
            display: flex;
            justify-content: space-around;
            padding: 8px 0;
        }
        
        .nav-item {
            text-align: center;
            text-decoration: none;
            color: var(--teal-gray);
            transition: var(--transition);
            padding: 5px;
            border-radius: 8px;
            min-width: 60px;
            position: relative;
        }
        
        .nav-item.active {
            color: var(--teal-primary);
        }
        
        .nav-item.active::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 50%;
            transform: translateX(-50%);
            width: 30px;
            height: 3px;
            background: var(--teal-primary);
            border-radius: 3px;
        }
        
        .nav-icon {
            font-size: 18px;
            margin-bottom: 2px;
        }
        
        .nav-label {
            font-size: 10px;
            font-weight: 500;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 30px 15px;
            color: var(--teal-text-medium);
        }
        
        .empty-icon {
            font-size: 48px;
            color: var(--teal-gray);
            margin-bottom: 15px;
        }
        
        .empty-text {
            font-size: 16px;
            margin-bottom: 10px;
        }
        
        .empty-subtext {
            font-size: 14px;
        }
        
        /* Disabled State */
        .disabled-overlay {
            position: relative;
        }
        
        .disabled-overlay::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255, 255, 255, 0.7);
            z-index: 10;
            cursor: not-allowed;
        }
    </style>
</head>
<body>
    <div class="app-container">
        <!-- Header Section -->
        <div class="app-header">
            <button class="back-btn" onclick="history.back()">
                <i class="fas fa-arrow-left"></i>
            </button>
            <h1 class="header-title">লোন</h1>
            <div class="balance-badge">
                <i class="fas fa-wallet"></i>
                ৳<?php echo number_format($userBalance, 2); ?>
            </div>
        </div>
        
        <!-- Balance Requirement -->
        <div class="balance-requirement">
            <div class="balance-requirement-text">
                <i class="fas <?php echo $hasEnoughBalance ? 'fa-check-circle' : 'fa-exclamation-triangle'; ?>"></i>
                <span>
                    <?php 
                    if ($hasEnoughBalance) {
                        echo "লোনের জন্য যোগ্যতা অর্জিত হয়েছে! আপনার অ্যাকাউন্ট ব্যালেন্স পর্যাপ্ত।";
                    } else {
                        echo "লোনের জন্য আপনার অ্যাকাউন্টে ন্যূনতম ৳$minRequiredBalance ব্যালেন্স প্রয়োজন। আপনার বর্তমান ব্যালেন্স: ৳$userBalance।";
                    }
                    ?>
                </span>
            </div>
            <div class="balance-progress">
                <div class="balance-progress-bar"></div>
            </div>
            <div class="balance-stats">
                <span class="balance-current">বর্তমান: ৳<?php echo number_format($userBalance, 2); ?></span>
                <span class="balance-required">প্রয়োজন: ৳<?php echo number_format($minRequiredBalance, 2); ?></span>
            </div>
        </div>
        
        <!-- Notice Container -->
        <div class="notice-container" id="noticeContainer">
            <div class="notice-text">
                <i class="fas fa-info-circle"></i>
                <span><?php echo htmlspecialchars($loanNotice); ?></span>
            </div>
            <button class="close-notice" onclick="closeNotice()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <!-- Balance Card -->
        <div class="balance-card">
            <div class="balance-greeting">
                <i class="fas fa-user-circle"></i>
                <span>হ্যালো, <?php echo $userName; ?>!</span>
            </div>
            <div class="balance-label">অ্যাকাউন্ট ব্যালেন্স</div>
            <div class="balance-amount">৳ <?php echo number_format($userBalance, 2); ?></div>
            <div class="reward-points">
                <i class="fas fa-coins"></i>
                রিওয়ার্ড পয়েন্ট: <?php echo number_format($rewardPoints, 0); ?>
            </div>
        </div>
        
        <!-- Form Card -->
        <div class="form-card <?php echo !$hasEnoughBalance ? 'disabled-overlay' : ''; ?>">
            <div class="form-card-header">
                <h2 class="form-title">নতুন লোনের আবেদন</h2>
            </div>
            <div class="form-card-body">
                <form id="loanForm" method="POST">
                    <!-- Loan Type Selection -->
                    <div class="loan-type-selection">
                        <label class="form-label">লোনের ধরন নির্বাচন করুন</label>
                        <div class="offers-grid">
                            <div class="offer-card" data-type="emergency" data-min="5000" data-max="50000" data-term="30 দিন">
                                <div class="offer-title">
                                    <i class="fas fa-hand-holding-usd"></i>
                                    ইমারজেন্সি লোন
                                </div>
                                <div class="offer-details">সর্বোচ্চ ৳৫০,০০০</div>
                                <div class="offer-details">মেয়াদ: ৩০ দিন</div>
                                <div class="offer-amount">৳৫,০০০ - ৳৫০,০০০</div>
                            </div>
                            <div class="offer-card" data-type="education" data-min="10000" data-max="100000" data-term="12 মাস">
                                <div class="offer-title">
                                    <i class="fas fa-graduation-cap"></i>
                                    শিক্ষা লোন
                                </div>
                                <div class="offer-details">সর্বোচ্চ ৳১,০০,০০০</div>
                                <div class="offer-details">মেয়াদ: ১২ মাস</div>
                                <div class="offer-amount">৳১০,০০০ - ৳১,০০,০০০</div>
                            </div>
                            <div class="offer-card" data-type="medical" data-min="2000" data-max="20000" data-term="60 দিন">
                                <div class="offer-title">
                                    <i class="fas fa-briefcase-medical"></i>
                                    মেডিকেল লোন
                                </div>
                                <div class="offer-details">সর্বোচ্চ ৳২০,০০০</div>
                                <div class="offer-details">মেয়াদ: ৬০ দিন</div>
                                <div class="offer-amount">৳২,০০০ - ৳২০,০০০</div>
                            </div>
                            <div class="offer-card" data-type="personal" data-min="20000" data-max="200000" data-term="24 মাস">
                                <div class="offer-title">
                                    <i class="fas fa-home"></i>
                                    পার্সোনাল লোন
                                </div>
                                <div class="offer-details">সর্বোচ্চ ৳২,০০,০০০</div>
                                <div class="offer-details">মেয়াদ: ২৪ মাস</div>
                                <div class="offer-amount">৳২০,০০০ - ৳২,০০,০০০</div>
                            </div>
                        </div>
                        
                        <!-- Loan Type Details -->
                        <div id="loanTypeDetails" class="loan-type-details">
                            <div class="loan-type-detail-item">
                                <span class="loan-type-detail-label">লোনের ধরন:</span>
                                <span class="loan-type-detail-value" id="selectedLoanType">-</span>
                            </div>
                            <div class="loan-type-detail-item">
                                <span class="loan-type-detail-label">ন্যূনতম পরিমাণ:</span>
                                <span class="loan-type-detail-value" id="selectedMinAmount">-</span>
                            </div>
                            <div class="loan-type-detail-item">
                                <span class="loan-type-detail-label">সর্বোচ্চ পরিমাণ:</span>
                                <span class="loan-type-detail-value" id="selectedMaxAmount">-</span>
                            </div>
                            <div class="loan-type-detail-item">
                                <span class="loan-type-detail-label">মেয়াদ:</span>
                                <span class="loan-type-detail-value" id="selectedTerm">-</span>
                            </div>
                            <div class="loan-type-detail-item">
                                <span class="loan-type-detail-label">সুদের হার:</span>
                                <span class="loan-type-detail-value"><?php echo $interestRate; ?>%</span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">লোনের পরিমাণ (৳)</label>
                        <div class="input-with-icon">
                            <i class="fas fa-money-bill-wave input-icon"></i>
                            <input type="number" name="loan_amount" id="loanAmount" class="form-control" placeholder="লোনের পরিমাণ লিখুন" min="<?php echo $minLoanAmount; ?>" max="<?php echo $maxLoanAmount; ?>" required <?php echo !$hasEnoughBalance ? 'disabled' : ''; ?>>
                        </div>
                        <small id="amountRange" style="color: var(--teal-gray); font-size: 12px;">ন্যূনতম: ৳<?php echo $minLoanAmount; ?>, সর্বোচ্চ: ৳<?php echo $maxLoanAmount; ?></small>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">লোনের উদ্দেশ্য</label>
                        <div class="input-with-icon">
                            <i class="fas fa-edit input-icon"></i>
                            <input type="text" name="loan_purpose" id="loanPurpose" class="form-control" placeholder="লোনের উদ্দেশ্য লিখুন (ঐচ্ছিক)" <?php echo !$hasEnoughBalance ? 'disabled' : ''; ?>>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">লোনের মেয়াদ</label>
                        <div class="select-wrapper">
                            <select name="loan_term" id="loanTerm" class="form-control" required <?php echo !$hasEnoughBalance ? 'disabled' : ''; ?>>
                                <option value="">মেয়াদ নির্বাচন করুন</option>
                                <option value="1">১ মাস</option>
                                <option value="3">৩ মাস</option>
                                <option value="6">৬ মাস</option>
                                <option value="12">১২ মাস</option>
                                <option value="24">২৪ মাস</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">লোনের তারিখ</label>
                        <div class="input-with-icon">
                            <i class="fas fa-calendar-alt input-icon"></i>
                            <input type="date" name="loan_date" id="loanDate" class="form-control" required <?php echo !$hasEnoughBalance ? 'disabled' : ''; ?>>
                        </div>
                    </div>
                    
                    <input type="hidden" id="pinInput" name="pin">
                    
                    <button type="button" class="submit-btn" onclick="showPopup()" <?php echo !$hasEnoughBalance ? 'disabled' : ''; ?>>
                        <i class="fas fa-paper-plane"></i>
                        আবেদন করুন
                    </button>
                </form>
            </div>
        </div>
        
        <!-- Existing Loans -->
        <div class="existing-loans">
            <h2 class="section-title">আপনার লোন সমূহ</h2>
            <?php if (count($existingLoans) > 0): ?>
                <div class="loan-list">
                    <?php foreach ($existingLoans as $loan): ?>
                        <div class="loan-item">
                            <div class="loan-header">
                                <div class="loan-id">লোন #<?php echo $loan['id']; ?></div>
                                <div class="loan-status status-<?php echo strtolower($loan['status']); ?>">
                                    <?php echo $loan['status']; ?>
                                </div>
                            </div>
                            <div class="loan-amount">৳<?php echo number_format($loan['amount'], 2); ?></div>
                            <div class="loan-purpose"><?php echo $loan['purpose'] ? $loan['purpose'] : 'উল্লেখ করা হয়নি'; ?></div>
                            <div class="loan-term">মেয়াদ: <?php echo $loan['term_months']; ?> মাস</div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <div class="empty-icon">
                        <i class="fas fa-hand-holding-usd"></i>
                    </div>
                    <div class="empty-text">কোনো লোন নেই</div>
                    <div class="empty-subtext">আপনি এখনো কোনো লোনের জন্য আবেদন করেননি</div>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Bottom Navigation -->
        <div class="bottom-nav">
            <div class="nav-list">
                <a href="/" class="nav-item">
                    <i class="fas fa-home nav-icon"></i>
                    <div class="nav-label">হোম</div>
                </a>
                <a href="history.php" class="nav-item">
                    <i class="fas fa-history nav-icon"></i>
                    <div class="nav-label">ইতিহাস</div>
                </a>
                <a href="coin.php" class="nav-item">
                    <i class="fas fa-qrcode nav-icon"></i>
                    <div class="nav-label">স্ক্যান</div>
                </a>
                <a href="coin.php" class="nav-item">
                    <img src="img/tt.png" alt="Rewards" style="width: 16px; height: 14px; margin-bottom: 1px;">
                    <div class="nav-label">রিওয়ার্ড</div>
                </a>
                <a href="profile.php" class="nav-item">
                    <i class="fas fa-user nav-icon"></i>
                    <div class="nav-label">প্রোফাইল</div>
                </a>
            </div>
        </div>
    </div>
    
    <!-- PIN Popup -->
    <div class="popup-overlay" id="popupOverlay">
        <div class="popup">
            <div class="popup-header">
                <h3 class="popup-title">পিন লিখুন</h3>
            </div>
            <div class="popup-body">
                <input type="password" id="pinEntry" class="pin-input" placeholder="••••" maxlength="4" required>
            </div>
            <div class="popup-footer">
                <button type="button" class="popup-btn cancel-btn" onclick="closePopup()">বাতিল</button>
                <button type="button" class="popup-btn verify-btn" onclick="submitForm()">যাচাই করুন</button>
            </div>
        </div>
    </div>
    
    <!-- Toast Notification -->
    <div id="toast" class="toast">
        <i class="toast-icon fas"></i>
        <span class="toast-message"></span>
    </div>
    
    <script>
        // Set today's date as the default value for loan date
        document.addEventListener('DOMContentLoaded', function() {
            const today = new Date().toISOString().split('T')[0];
            document.getElementById('loanDate').value = today;
            
            const offerCards = document.querySelectorAll('.offer-card');
            const loanTypeDetails = document.getElementById('loanTypeDetails');
            const selectedLoanType = document.getElementById('selectedLoanType');
            const selectedMinAmount = document.getElementById('selectedMinAmount');
            const selectedMaxAmount = document.getElementById('selectedMaxAmount');
            const selectedTerm = document.getElementById('selectedTerm');
            const loanAmountInput = document.getElementById('loanAmount');
            const loanPurposeInput = document.getElementById('loanPurpose');
            const loanTermSelect = document.getElementById('loanTerm');
            const amountRange = document.getElementById('amountRange');
            
            // Check if user has enough balance
            const hasEnoughBalance = <?php echo $hasEnoughBalance ? 'true' : 'false'; ?>;
            
            // Get loan type names
            const loanTypeNames = {
                'emergency': 'ইমারজেন্সি লোন',
                'education': 'শিক্ষা লোন',
                'medical': 'মেডিকেল লোন',
                'personal': 'পার্সোনাল লোন'
            };
            
            // Add click event to each offer card
            offerCards.forEach(card => {
                card.addEventListener('click', function() {
                    if (!hasEnoughBalance) {
                        showToast('লোনের জন্য আপনার অ্যাকাউন্ট ব্যালেন্স পর্যাপ্ত নয়', 'error');
                        return;
                    }
                    
                    // Remove selected class from all cards
                    offerCards.forEach(c => c.classList.remove('selected'));
                    
                    // Add selected class to clicked card
                    this.classList.add('selected');
                    
                    // Get loan type data
                    const loanType = this.getAttribute('data-type');
                    const minAmount = this.getAttribute('data-min');
                    const maxAmount = this.getAttribute('data-max');
                    const term = this.getAttribute('data-term');
                    
                    // Update loan type details
                    selectedLoanType.textContent = loanTypeNames[loanType];
                    selectedMinAmount.textContent = '৳' + parseInt(minAmount).toLocaleString('bn-BD');
                    selectedMaxAmount.textContent = '৳' + parseInt(maxAmount).toLocaleString('bn-BD');
                    selectedTerm.textContent = term;
                    
                    // Show loan type details
                    loanTypeDetails.classList.add('active');
                    
                    // Auto-fill loan amount with minimum value
                    loanAmountInput.value = minAmount;
                    loanAmountInput.min = minAmount;
                    loanAmountInput.max = maxAmount;
                    
                    // Update amount range text
                    amountRange.textContent = `ন্যূনতম: ৳${parseInt(minAmount).toLocaleString('bn-BD')}, সর্বোচ্চ: ৳${parseInt(maxAmount).toLocaleString('bn-BD')}`;
                    
                    // Auto-fill loan purpose with loan type name
                    loanPurposeInput.value = loanTypeNames[loanType];
                    
                    // Auto-select loan term based on selected type
                    if (term.includes('দিন')) {
                        // For short-term loans, select 1 month
                        loanTermSelect.value = '1';
                    } else if (term.includes('মাস')) {
                        const months = parseInt(term);
                        if (months <= 24) {
                            loanTermSelect.value = months.toString();
                        } else {
                            loanTermSelect.value = '24';
                        }
                    }
                });
            });
            
            // Add event listener for amount input to update details dynamically
            loanAmountInput.addEventListener('input', function() {
                if (loanTypeDetails.classList.contains('active')) {
                    // Update the selected amount in the details
                    const amountValue = this.value;
                    if (amountValue) {
                        // Calculate interest for display
                        const interestRate = <?php echo $interestRate; ?>;
                        const interestAmount = (amountValue * interestRate) / 100;
                        const totalRepayment = parseFloat(amountValue) + interestAmount;
                        
                        // Update or create the repayment detail item
                        let repaymentItem = document.getElementById('repaymentDetail');
                        if (!repaymentItem) {
                            repaymentItem = document.createElement('div');
                            repaymentItem.className = 'loan-type-detail-item';
                            repaymentItem.id = 'repaymentDetail';
                            loanTypeDetails.appendChild(repaymentItem);
                        }
                        
                        repaymentItem.innerHTML = `
                            <span class="loan-type-detail-label">মোট পরিশোধ:</span>
                            <span class="loan-type-detail-value">৳${parseFloat(totalRepayment).toLocaleString('bn-BD')} (${interestRate}% সুদসহ)</span>
                        `;
                    }
                }
            });
        });
        
        // Close notice container
        function closeNotice() {
            const noticeContainer = document.getElementById('noticeContainer');
            noticeContainer.style.display = 'none';
        }
        
        // Show PIN popup
        function showPopup() {
            const hasEnoughBalance = <?php echo $hasEnoughBalance ? 'true' : 'false'; ?>;
            
            if (!hasEnoughBalance) {
                showToast('লোনের জন্য আপনার অ্যাকাউন্ট ব্যালেন্স পর্যাপ্ত নয়', 'error');
                return;
            }
            
            const form = document.getElementById('loanForm');
            const selectedCard = document.querySelector('.offer-card.selected');
            
            if (!selectedCard) {
                showToast('অনুগ্রহ করে লোনের ধরন নির্বাচন করুন', 'error');
                return;
            }
            
            if (form.checkValidity()) {
                const popupOverlay = document.getElementById('popupOverlay');
                popupOverlay.classList.add('show');
                document.getElementById('pinEntry').value = '';
                document.getElementById('pinEntry').focus();
            } else {
                form.reportValidity();
            }
        }
        
        // Close PIN popup
        function closePopup() {
            const popupOverlay = document.getElementById('popupOverlay');
            popupOverlay.classList.remove('show');
        }
        
        // Verify payment and submit the form
        function submitForm() {
            const enteredPin = document.getElementById('pinEntry').value;
            if (enteredPin.length === 4) {
                document.getElementById('pinInput').value = enteredPin;
                document.getElementById('loanForm').submit();
            } else {
                showToast('অনুগ্রহ করে ৪ ডিজিট পিন লিখুন', 'error');
            }
        }
        
        // Show toast notification
        function showToast(message, type = 'success') {
            const toast = document.getElementById('toast');
            const toastMessage = toast.querySelector('.toast-message');
            const toastIcon = toast.querySelector('.toast-icon');
            
            // Set toast content
            toastMessage.textContent = message;
            
            // Set toast type
            toast.className = 'toast';
            toast.classList.add(type);
            
            // Set icon based on type
            if (type === 'success') {
                toastIcon.className = 'toast-icon fas fa-check-circle';
            } else if (type === 'error') {
                toastIcon.className = 'toast-icon fas fa-exclamation-circle';
            }
            
            // Show toast
            toast.classList.add('show');
            
            // Hide toast after 3 seconds
            setTimeout(() => {
                toast.classList.remove('show');
            }, 3000);
        }
        
        // Show toast notification based on PHP response
        <?php if ($successMessage): ?>
            showToast('<?php echo $successMessage; ?>', 'success');
            setTimeout(function() {
                window.location.href = "dashboard.php";
            }, 2000);
        <?php elseif ($showErrorToast): ?>
            showToast('<?php echo $loanError; ?>', 'error');
        <?php endif; ?>
        
        // Close popup when clicking outside
        document.getElementById('popupOverlay').addEventListener('click', function(e) {
            if (e.target === this) {
                closePopup();
            }
        });
    </script>
</body>
</html>