<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include the database connection
include('db.php');

// Check database connection
if (!$conn) {
    die("Database connection failed: " . mysqli_connect_error());
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Update loan status
    if (isset($_POST['update_loan_status'])) {
        $loan_id = $_POST['loan_id'];
        $status = $_POST['status'];
        
        $updateQuery = $conn->prepare("UPDATE loans SET status = ? WHERE id = ?");
        $updateQuery->bind_param("si", $status, $loan_id);
        $updateQuery->execute();
        
        $success_message = "Loan status updated successfully!";
    }
    
    // Update settings
    if (isset($_POST['update_settings'])) {
        $min_loan = $_POST['min_loan_amount'];
        $max_loan = $_POST['max_loan_amount'];
        $interest = $_POST['interest_rate'];
        $notice = $_POST['loan_notice'];
        
        $updateQuery = $conn->prepare("UPDATE settings SET min_loan_amount = ?, max_loan_amount = ?, interest_rate = ?, loan_notice = ? WHERE id = 1");
        $updateQuery->bind_param("ddss", $min_loan, $max_loan, $interest, $notice);
        $updateQuery->execute();
        
        $success_message = "Settings updated successfully!";
    }
    
    // Delete loan
    if (isset($_POST['delete_loan'])) {
        $loan_id = $_POST['loan_id'];
        
        $deleteQuery = $conn->prepare("DELETE FROM loans WHERE id = ?");
        $deleteQuery->bind_param("i", $loan_id);
        $deleteQuery->execute();
        
        $success_message = "Loan deleted successfully!";
    }
}

// Fetch all loans with user information
$loansQuery = "SELECT l.*, u.name as user_name FROM loans l JOIN users u ON l.user_id = u.id ORDER BY l.created_at DESC";
$loansResult = $conn->query($loansQuery);
$loans = $loansResult->fetch_all(MYSQLI_ASSOC);

// Fetch settings
$settingsQuery = "SELECT * FROM settings WHERE id = 1";
$settingsResult = $conn->query($settingsQuery);
$settings = $settingsResult->fetch_assoc();

// Fetch users - FIXED: Removed email column
$usersQuery = "SELECT id, name, balance, created_at FROM users ORDER BY created_at DESC";
$usersResult = $conn->query($usersQuery);
$users = $usersResult->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Control Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Hind+Siliguri:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Hind Siliguri', sans-serif;
        }
        
        :root {
            --teal-primary: #008080;
            --teal-light: #e0fffe;
            --teal-dark: #004d40;
            --teal-medium: #4dbdbd;
            --teal-text-dark: #004d40;
            --teal-text-medium: #00695c;
            --teal-text-light: #00a39a;
            --teal-gray-lightest: #f5fafa;
            --teal-gray-light: #e0f2f1;
            --teal-gray: #80cbc4;
            --teal-gray-dark: #00695c;
            --teal-border: #b2dfdb;
            --teal-success: #2ecc71;
            --teal-warning: #f39c12;
            --teal-danger: #e74c3c;
            --teal-blue: #3498db;
            --transition: all 0.3s ease;
        }
        
        body {
            background: var(--teal-gray-lightest);
            color: var(--teal-text-dark);
            font-size: 14px;
            line-height: 1.6;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            background: linear-gradient(135deg, var(--teal-primary), var(--teal-dark));
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            box-shadow: 0 4px 12px rgba(0, 128, 128, 0.15);
        }
        
        .header h1 {
            font-size: 24px;
            margin-bottom: 5px;
        }
        
        .header p {
            opacity: 0.9;
        }
        
        .dashboard {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            text-align: center;
        }
        
        .stat-card i {
            font-size: 32px;
            margin-bottom: 10px;
        }
        
        .stat-card.total-users i { color: var(--teal-primary); }
        .stat-card.total-loans i { color: var(--teal-blue); }
        .stat-card.pending-loans i { color: var(--teal-warning); }
        .stat-card.total-balance i { color: var(--teal-success); }
        
        .stat-card h3 {
            font-size: 28px;
            margin-bottom: 5px;
        }
        
        .stat-card p {
            color: var(--teal-text-medium);
        }
        
        .tabs {
            display: flex;
            margin-bottom: 20px;
            border-bottom: 1px solid var(--teal-border);
        }
        
        .tab {
            padding: 12px 20px;
            cursor: pointer;
            border-bottom: 3px solid transparent;
            transition: var(--transition);
        }
        
        .tab.active {
            color: var(--teal-primary);
            border-bottom-color: var(--teal-primary);
        }
        
        .tab:hover {
            color: var(--teal-primary);
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .table-container {
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            margin-bottom: 20px;
        }
        
        .table-header {
            background: var(--teal-gray-light);
            padding: 15px 20px;
            font-weight: 600;
            color: var(--teal-text-dark);
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th, td {
            padding: 12px 15px;
            text-align: left;
            border-bottom: 1px solid var(--teal-gray-light);
        }
        
        th {
            background: var(--teal-gray-lightest);
            font-weight: 600;
            color: var(--teal-text-dark);
        }
        
        tr:hover {
            background: var(--teal-gray-lightest);
        }
        
        .status-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .status-pending { background: var(--teal-warning); color: white; }
        .status-approved { background: var(--teal-success); color: white; }
        .status-rejected { background: var(--teal-danger); color: white; }
        
        .form-container {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            margin-bottom: 20px;
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
            color: var(--teal-text-dark);
        }
        
        .form-control {
            width: 100%;
            padding: 10px;
            border: 1px solid var(--teal-border);
            border-radius: 5px;
            font-size: 14px;
        }
        
        textarea.form-control {
            min-height: 100px;
            resize: vertical;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-weight: 500;
            transition: var(--transition);
        }
        
        .btn-primary {
            background: var(--teal-primary);
            color: white;
        }
        
        .btn-primary:hover {
            background: var(--teal-dark);
        }
        
        .btn-danger {
            background: var(--teal-danger);
            color: white;
        }
        
        .btn-danger:hover {
            background: #c0392b;
        }
        
        .btn-sm {
            padding: 5px 10px;
            font-size: 12px;
        }
        
        .alert {
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        
        .alert-success {
            background: var(--teal-success);
            color: white;
        }
        
        @media (max-width: 768px) {
            .container { padding: 10px; }
            .dashboard { grid-template-columns: 1fr; }
            .tabs { overflow-x: auto; white-space: nowrap; }
            .table-container { overflow-x: auto; }
            table { min-width: 600px; }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-cogs"></i> Admin Control Panel</h1>
            <p>Ramit Wallet Management System</p>
        </div>
        
        <?php if (isset($success_message)): ?>
            <div class="alert alert-success">
                <?php echo $success_message; ?>
            </div>
        <?php endif; ?>
        
        <div class="dashboard">
            <div class="stat-card total-users">
                <i class="fas fa-users"></i>
                <h3><?php echo count($users); ?></h3>
                <p>Total Users</p>
            </div>
            <div class="stat-card total-loans">
                <i class="fas fa-hand-holding-usd"></i>
                <h3><?php echo count($loans); ?></h3>
                <p>Total Loans</p>
            </div>
            <div class="stat-card pending-loans">
                <i class="fas fa-clock"></i>
                <h3>
                    <?php 
                    $pending_count = 0;
                    foreach ($loans as $loan) {
                        if ($loan['status'] === 'Pending') $pending_count++;
                    }
                    echo $pending_count;
                    ?>
                </h3>
                <p>Pending Loans</p>
            </div>
            <div class="stat-card total-balance">
                <i class="fas fa-wallet"></i>
                <h3>
                    <?php 
                    $total_balance = 0;
                    foreach ($users as $user) {
                        $total_balance += $user['balance'];
                    }
                    echo '৳' . number_format($total_balance, 2);
                    ?>
                </h3>
                <p>Total Balance</p>
            </div>
        </div>
        
        <div class="tabs">
            <div class="tab active" onclick="showTab('loans')">Loans Management</div>
            <div class="tab" onclick="showTab('users')">Users Management</div>
            <div class="tab" onclick="showTab('settings')">Settings</div>
        </div>
        
        <div id="loans-tab" class="tab-content active">
            <div class="table-container">
                <div class="table-header">
                    <i class="fas fa-hand-holding-usd"></i> Loans Management
                </div>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>User</th>
                            <th>Amount</th>
                            <th>Purpose</th>
                            <th>Term</th>
                            <th>Status</th>
                            <th>Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($loans as $loan): ?>
                            <tr>
                                <td><?php echo $loan['id']; ?></td>
                                <td><?php echo $loan['user_name']; ?></td>
                                <td>৳<?php echo number_format($loan['amount'], 2); ?></td>
                                <td><?php echo $loan['purpose'] ?: 'N/A'; ?></td>
                                <td><?php echo $loan['term_months']; ?> months</td>
                                <td>
                                    <span class="status-badge status-<?php echo strtolower($loan['status']); ?>">
                                        <?php echo $loan['status']; ?>
                                    </span>
                                </td>
                                <td><?php echo date('d M Y', strtotime($loan['created_at'])); ?></td>
                                <td>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="loan_id" value="<?php echo $loan['id']; ?>">
                                        <select name="status" class="form-control" style="width: auto; display: inline-block;">
                                            <option value="Pending" <?php echo $loan['status'] === 'Pending' ? 'selected' : ''; ?>>Pending</option>
                                            <option value="Approved" <?php echo $loan['status'] === 'Approved' ? 'selected' : ''; ?>>Approved</option>
                                            <option value="Rejected" <?php echo $loan['status'] === 'Rejected' ? 'selected' : ''; ?>>Rejected</option>
                                        </select>
                                        <button type="submit" name="update_loan_status" class="btn btn-primary btn-sm">Update</button>
                                    </form>
                                    <form method="POST" style="display: inline; margin-left: 5px;">
                                        <input type="hidden" name="loan_id" value="<?php echo $loan['id']; ?>">
                                        <button type="submit" name="delete_loan" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure?')">Delete</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <div id="users-tab" class="tab-content">
            <div class="table-container">
                <div class="table-header">
                    <i class="fas fa-users"></i> Users Management
                </div>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Balance</th>
                            <th>Joined</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                            <tr>
                                <td><?php echo $user['id']; ?></td>
                                <td><?php echo $user['name']; ?></td>
                                <td>৳<?php echo number_format($user['balance'], 2); ?></td>
                                <td><?php echo date('d M Y', strtotime($user['created_at'])); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <div id="settings-tab" class="tab-content">
            <div class="form-container">
                <div class="table-header">
                    <i class="fas fa-cog"></i> Loan Settings
                </div>
                <form method="POST">
                    <div class="form-group">
                        <label>Minimum Loan Amount (৳)</label>
                        <input type="number" name="min_loan_amount" class="form-control" value="<?php echo $settings['min_loan_amount']; ?>" required>
                    </div>
                    <div class="form-group">
                        <label>Maximum Loan Amount (৳)</label>
                        <input type="number" name="max_loan_amount" class="form-control" value="<?php echo $settings['max_loan_amount']; ?>" required>
                    </div>
                    <div class="form-group">
                        <label>Interest Rate (%)</label>
                        <input type="number" step="0.01" name="interest_rate" class="form-control" value="<?php echo $settings['interest_rate']; ?>" required>
                    </div>
                    <div class="form-group">
                        <label>Loan Notice</label>
                        <textarea name="loan_notice" class="form-control"><?php echo $settings['loan_notice']; ?></textarea>
                    </div>
                    <button type="submit" name="update_settings" class="btn btn-primary">Update Settings</button>
                </form>
            </div>
        </div>
    </div>
    
    <script>
        function showTab(tabName) {
            const tabs = document.querySelectorAll('.tab-content');
            tabs.forEach(tab => {
                tab.classList.remove('active');
            });
            
            const tabButtons = document.querySelectorAll('.tab');
            tabButtons.forEach(button => {
                button.classList.remove('active');
            });
            
            document.getElementById(tabName + '-tab').classList.add('active');
            event.target.classList.add('active');
        }
    </script>
</body>
</html>